import logging
import os
import requests
import pytz
from argparse import ArgumentParser
from collections import namedtuple
from datetime import datetime, timedelta
from functools import partial
from typing import List

from travel.avia.flight_extras.application import db
from travel.avia.flight_extras.application.service.sky_guru import sky_guru
from travel.avia.flight_extras.application.models import Flight, FlightInfo
from travel.avia.flight_extras.settings import FLIGHT_STORAGE_API_URL, SKY_GURU_FETCH_LIMIT, setup_logging

logger = logging.getLogger('fetch_sky_guru')


def fetch_flight(company_iata, number, departure):
    # type: (str, str, datetime) -> None
    flight = sky_guru.get_flight(
        company_iata,
        number,
        departure
    )

    if not flight:
        logger.info('No info in sky guru for flight "%s %s" at "%s"', company_iata, number, departure)
        return

    route = sky_guru.get_route(flight['id'])
    sun = sky_guru.get_sun(flight['id'])
    wind = sky_guru.get_wind(flight['id'])
    weather = sky_guru.get_weather(flight['id'])
    sights = sky_guru.get_sights(flight['id'])

    aircraft = sky_guru.get_aircraft(flight['aircraft_id'])
    if aircraft:
        aircraft.update(sky_guru.get_aircraft_extra_info(flight['aircraft_id']))

    airport_from = sky_guru.get_airport(flight['origin']['id'])
    airport_from.update({
        'id': flight['destination']['id'],
        'terminal': flight['destination']['terminal'],
        'gate': flight['destination']['gate'],
        'baggage': flight['destination']['baggage'],
    })

    airport_to = sky_guru.get_airport(flight['destination']['id'])
    airport_to.update({
        'id': flight['destination']['id'],
        'terminal': flight['destination']['terminal'],
        'gate': flight['destination']['gate'],
        'baggage': flight['destination']['baggage'],
    })

    session = db.db_master.create_session()
    f = session.query(Flight).filter_by(
        company_iata=company_iata,
        number=number,
    ).one_or_none()

    if not f:
        f = Flight(company_iata=company_iata, number=number)
        session.add(f)

    flight_info = session.query(FlightInfo).filter_by(
        flight_id=f.id,
        departure_day=departure.date(),
    ).one_or_none()

    if not flight_info:
        flight_info = FlightInfo(
            flight=f,
            departure_day=departure.date(),
        )
        session.add(flight_info)

    flight_info.turbulence_index = flight['turbulence_index']
    flight_info.direct_distance = flight['direct_distance']
    flight_info.distance = flight['distance']
    flight_info.average_speed = flight['average_speed']
    flight_info.duration = flight['duration']
    flight_info.sun = sun
    flight_info.wind = wind
    flight_info.sights = sights
    flight_info.aircraft = aircraft
    flight_info.airport_from = airport_from
    flight_info.airport_to = airport_to

    if route:
        flight_info.turbulence_zones = route.get('zones', None)
        flight_info.route = route.get('points', None)

    if weather:
        flight_info.weather_from = weather.get('origin', None)
        flight_info.weather_to = weather.get('destination', None)

    session.commit()


def get_flights(ids, filters=None):
    # type: (list, list) -> list[dict]
    response = requests.get(FLIGHT_STORAGE_API_URL + '/get/flight/', params={
        'flight': ','.join(ids),
    })
    response.raise_for_status()

    flights = response.json()

    if filters:
        for f in filters:
            flights = filter(f, flights)

    return flights


def filter_departure(flight, td=timedelta(days=1)):
    # type: (dict, timedelta) -> bool
    d = datetime.strptime(flight['departure_utc'], '%Y-%m-%d %H:%M:%S')
    now = datetime.utcnow()
    t = now + td

    return now <= d <= t


def filter_fetched(flight):
    # type: (dict) -> bool
    company_iata, number = flight['number'].split(' ')
    f = db.db_slave.create_session().query(FlightInfo).join(Flight)\
        .filter(Flight.company_iata == company_iata)\
        .filter(Flight.number == number)\
        .filter(FlightInfo.departure_day == flight['departure_day'])\
        .one_or_none()

    return f is None


def fetch_sky_guru(departure_day, filters=None, limit=None, flights_source=None):
    # type: (datetime, list, int, List[str]) -> int
    logger.info('Fetch on %s', departure_day.strftime('%Y-%m-%d'))

    if flights_source is None:
        flights_source = db.db_slave.create_session().query(Flight).all()

    flights = []
    ids = []
    for f in flights_source:
        ids.append(departure_day.strftime('%Y%m%d') + f.company_iata + f.number)
        if len(ids) == 1000:
            flights = flights + get_flights(ids, filters)
            ids = []
        if limit and len(flights) >= limit:
            break

    if len(ids) > 0:
        flights = flights + get_flights(ids, filters)

    if limit:
        flights = flights[:limit]

    for f in flights:
        departure = pytz.timezone(f['departure_timezone']).localize(datetime.strptime(
            f['departure_day'] + ' ' + f['departure_time'],
            '%Y-%m-%d %H:%M:%S',
        ))

        company_iata, number = f['number'].split(' ')
        fetch_flight(company_iata, number, departure)

    return len(flights)


def main():
    setup_logging()

    if os.getenv('YANDEX_ENVIRONMENT_TYPE') not in ('development', 'testing', 'production'):
        logger.info('Allow only in %s', ('development', 'testing', 'production'))
        return

    fetch_limit = SKY_GURU_FETCH_LIMIT
    logger.info('Fetch limit: %d', fetch_limit)

    flights = []
    if args.data_file:
        logger.info('Fetch flights from file: %s', args.data_file)
        with open(args.data_file, 'r') as f:
            FlightNamedTuple = namedtuple('FlightNamedTuple', 'company_iata number')
            for s in f.readlines():
                company_iata, number = s.strip().split(' ')
                flights.append(FlightNamedTuple(company_iata=company_iata, number=number))

    fetched = fetch_sky_guru(datetime.now() + timedelta(days=1), [
        filter_departure,
        filter_fetched,
    ], fetch_limit, flights)

    if fetch_limit - fetched > 0:
        fetch_sky_guru(datetime.now(), [
            partial(filter_departure, td=timedelta(hours=3)),
            filter_departure,
        ], fetch_limit - fetched, flights)


if __name__ == '__main__':
    argparser = ArgumentParser()
    argparser.add_argument('-d', '--data-file', dest='data_file', required=False)
    args = argparser.parse_args()

    main()
