import argparse
import logging
import time

from requests.exceptions import ConnectionError

from travel.avia.library.python.iata_correction import IATACorrector
from travel.avia.library.python.shared_dicts.cache.company_cache import CompanyCache

from travel.avia.flight_status_fetcher.library.logging import setup_logging
from travel.avia.flight_status_fetcher.library.airport_updater import AirportUpdater
from travel.avia.flight_status_fetcher.library.flight_number_parser import FlightNumberParser
from travel.avia.flight_status_fetcher.library.solomon import transform_dict_to_sensors, upload_to_solomon
from travel.avia.flight_status_fetcher.settings.services.shared_flights import SHARED_FLIGHTS_API_BASE_URL

RESTART_EVERY = 60


def main():
    setup_logging()
    parser = argparse.ArgumentParser()
    parser.add_argument('airport', help='Airport code')

    program_args = parser.parse_args()

    airport = str(program_args.airport).lower()

    logger = logging.getLogger(__name__ + airport)

    iata_corrector = IATACorrector(
        SHARED_FLIGHTS_API_BASE_URL,
        logger,
    )

    companies = CompanyCache(logger)
    companies.populate()

    flight_number_parser = FlightNumberParser(companies, iata_corrector)

    try:
        updater = AirportUpdater(logger, flight_number_parser)
        for _ in range(RESTART_EVERY):  # перезапускать воркер примерно раз в час
            try:
                start = time.time()
                statistics = updater.update(airport)
                end = time.time()

                data = dict(statistics, elapsed=end - start)
                upload_to_solomon(
                    {
                        'source': airport,
                    },
                    transform_dict_to_sensors(end, data),
                )
                to_sleep = max(0.0, start + 60 - time.time())
                logger.info('Sleep for %f', to_sleep)
                if to_sleep:
                    time.sleep(to_sleep)
            except ConnectionError:
                logger.exception('Connection error to %s', airport)

    except AttributeError as e:
        logger.exception('Airport {} not found'.format(airport))
        raise e
    except Exception as e:
        logger.exception('Unexpected error %s: %s', airport, e)
        raise e


if __name__ == '__main__':
    main()
