import datetime
import ftplib
import logging
import urllib.parse
import uuid
from typing import Dict, Optional

import pytz

from retrying import retry

from travel.avia.library.python.proxy_pool.proxy_pool import ProxyPool

logger = logging.getLogger(__name__)


class FTPWrapper:
    def __init__(self, host, user, passwd, timeout, retry_kwargs: Dict, proxy_pool: Optional[ProxyPool] = None):
        self._host = host
        self._user = user
        self._passwd = passwd
        self._retry_kwargs = retry_kwargs
        self._timeout = timeout
        self._proxy_pool = proxy_pool

    def get_modification_time(self, filename: str) -> datetime.datetime:
        ftp = self._create_ftp()
        response = ftp.voidcmd('MDTM {}'.format(filename))
        ftp.quit()
        logger.info('MDTM response: %s', response)
        parts = response.split(' ', 1)
        if len(parts) != 2:
            raise ValueError('Bad response from MDTM: {}'.format(response))

        # Here we assume, that ftp server returns a timesampt in UTC
        return pytz.UTC.localize(datetime.datetime.strptime(parts[1], '%Y%m%d%H%M%S'))

    def retrieve_string(self, filename: str, encoding: str = 'utf-8'):
        request_id = str(uuid.uuid4())
        logger.info('Retrieving file %s from FTP at %s. request #%s', filename, self._host, request_id)
        quoted_filename = urllib.parse.quote(filename)
        blocks = []

        def _save_block(block: bytes):
            blocks.append(block)

        def _try_download_file():
            blocks.clear()
            logger.info('Trying to connect to FTP. request #%s', request_id)
            ftp = self._create_ftp()
            logger.info('Trying to download file from FTP. request #%s', request_id)
            ftp.retrbinary('RETR {}'.format(quoted_filename), callback=_save_block)
            logger.info('Closing conenction to FTP. request #%s', request_id)
            ftp.quit()

        @retry(
            retry_on_exception=lambda exc: isinstance(exc, (ftplib.Error, TimeoutError, OSError)),
            **self._retry_kwargs,
        )
        def _download_file():
            try:
                _try_download_file()
            except Exception as e:
                logger.info('FTP download failed: %s. Request #%s', e, request_id)
                raise

        _download_file()
        logger.info('FTP download complete. request #%s', request_id)
        return b''.join(blocks).decode(encoding)

    def _create_ftp(self):
        if self._proxy_pool:
            proxy = self._proxy_pool.get_proxy()
            ftp = ftplib.FTP(
                timeout=self._timeout,
            )
            ftp.connect(
                host=proxy.get_ftp_host(),
                port=proxy.FTP_PORT,
            )
            ftp.login(
                user=proxy.get_ftp_user(self._user, self._host),
                passwd=self._passwd,
            )
        else:
            ftp = ftplib.FTP(
                host=self._host,
                user=self._user,
                passwd=self._passwd,
                timeout=self._timeout,
            )

        return ftp
