import logging
from abc import ABC, abstractmethod
from typing import Any, Dict, Optional

from urllib3.util.retry import Retry

import requests
from requests.adapters import HTTPAdapter

from travel.avia.library.python.proxy_pool.proxy_pool import ProxyPool

from travel.avia.flight_status_fetcher.library.flight_number_parser import FlightNumberParser
from travel.avia.flight_status_fetcher.library.raw_data import StatusDataPack, StatusDataCollector
from travel.avia.flight_status_fetcher.library.statistics import ObjectWithStatistics
from travel.avia.flight_status_fetcher.utils.import_helper import ImportHelper


class AirportImporter(ImportHelper, ObjectWithStatistics, ABC):
    SOURCE = 'airport'

    def __init__(
        self,
        logger: logging.Logger,
        flight_number_parser: FlightNumberParser,
        *args,
        **kwargs,
    ):
        super().__init__(logger, flight_number_parser, *args, **kwargs)
        self.logger = logger
        self.status_data_collector = StatusDataCollector()

    @abstractmethod
    def collect_statuses(self, *args, **kwargs) -> StatusDataPack:
        pass


class AirportHTTPImporter(AirportImporter, ABC):
    def __init__(
        self,
        retry_settings: Dict[Any, Any],
        logger: logging.Logger,
        flight_number_parser: FlightNumberParser,
        proxy_pool: Optional[ProxyPool] = None,
        *args,
        **kwargs,
    ):
        super().__init__(logger, flight_number_parser, *args, **kwargs)

        self._proxy_pool = proxy_pool
        self._session = self._construct_session(retry_settings)

    @staticmethod
    def _construct_session(retry_settings):
        session = requests.Session()
        retry = Retry(**retry_settings)
        adapter = HTTPAdapter(max_retries=retry)
        session.mount('http://', adapter)
        session.mount('https://', adapter)

        return session

    def _curl(self, url, **kwargs):
        proxies = None
        if self._proxy_pool:
            proxies = self._proxy_pool.get_proxy().get_requests_proxies()
        response = self._session.get(
            url,
            proxies=proxies,
            **kwargs,
        )
        response.raise_for_status()
        self._last_run_statistics['datasize'] = len(response.content)
        return response.content
