import datetime
import logging
from functools import partial

from travel.avia.flight_status_fetcher.library.airport_importer import AirportImporter
from travel.avia.flight_status_fetcher.library.flight_number_parser import FlightNumberParser
from travel.avia.flight_status_fetcher.library.raw_data import StatusDataPack
from travel.avia.flight_status_fetcher.sources.basel import BaselImporter
from travel.avia.flight_status_fetcher.sources.dme import DMEImporter
from travel.avia.flight_status_fetcher.sources.led import LEDImporter
from travel.avia.flight_status_fetcher.sources.region_airports import RegionAirportImporter
from travel.avia.flight_status_fetcher.sources.svo import SVOFetcher, SVOImporter
from travel.avia.flight_status_fetcher.sources.vko import VKOImporter
from travel.avia.flight_status_fetcher.updaters import update_statuses, send_raw_data
from travel.avia.flight_status_fetcher.utils.wsdl import build_client
from travel.avia.flight_status_fetcher.settings import app
from travel.avia.flight_status_fetcher.settings.sources import basel as basel_settings, dme, led, region, svo, vko
from travel.avia.flight_status_fetcher.settings.services import proxy_pool


class AirportUpdater:
    def __init__(self, logger, flight_number_parser: FlightNumberParser):
        self._logger = logger
        self._flight_number_parser = flight_number_parser

    def vko_updater(self):
        return VKOImporter(
            vko.VKO_DATA_URL,
            vko.VKO_DATAFILENAME,
            vko.VKO_USER,
            vko.VKO_PASSWORD,
            logger=logging.getLogger('vko_updater'),
            flight_number_parser=self._flight_number_parser,
            retry_settings=app.AIRPORT_IMPORTER_RETRY_SETTINGS,
            request_timeout=vko.VKO_REQUEST_TIMEOUT,
            proxy_pool=proxy_pool.PARTNERS_PROXY_POOL,
        )

    def svo_updater(self, days_before=1, days_after=2):
        from_date = datetime.datetime.today() - datetime.timedelta(days=days_before)
        to_date = datetime.datetime.today() + datetime.timedelta(days=days_after)

        logger = logging.getLogger('svo_updater')
        svo_fetcher = SVOFetcher(
            svo.SVO_URL,
            svo.SVO_OAUTH_TOKEN,
            logger,
            proxy_pool=proxy_pool.PARTNERS_PROXY_POOL,
        )
        return SVOImporter(
            from_date=from_date,
            to_date=to_date,
            fetcher=svo_fetcher,
            logger=logger,
            flight_number_parser=self._flight_number_parser,
        )

    def dme_updater(self):
        return DMEImporter(
            dme.DME_DATAURL,
            logger=logging.getLogger('dme_updater'),
            flight_number_parser=self._flight_number_parser,
            retry_settings=app.AIRPORT_IMPORTER_RETRY_SETTINGS,
            request_timeout=dme.DME_REQUEST_TIMEOUT,
            proxy_pool=None,  # Без прокси, т.к. ходим через админку распов
        )

    def led_updater(self):
        return LEDImporter(
            led.LED_FTP_HOST,
            led.LED_FTP_USER,
            led.LED_FTP_PASSWORD,
            led.LED_ARRIVAL_DATA_FILENAME,
            led.LED_DEPARTURE_DATA_FILENAME,
            led.LED_RETRY_SETTINGS,
            timeout=led.LED_TIMEOUT,
            logger=logging.getLogger('led_updater'),
            flight_number_parser=self._flight_number_parser,
            proxy_pool=proxy_pool.PARTNERS_PROXY_POOL,
        )

    def region_airports_updater(self, iata):
        return RegionAirportImporter(
            iata=iata,
            retry_settings=app.AIRPORT_IMPORTER_RETRY_SETTINGS,
            request_timeout=region.REGION_AIRPORT_TIMEOUT,
            logger=logging.getLogger('region_airport_importer'),
            flight_number_parser=self._flight_number_parser,
            iata_to_url=region.REGION_AIRPORT_IATA,
            proxy_pool=proxy_pool.PARTNERS_PROXY_POOL,
        )

    def basel_updater(self, iata):
        return BaselImporter(
            iata=iata,
            wsdl_client=build_client(
                basel_settings.BASEL_URL,
                basel_settings.BASEL_LOGIN,
                basel_settings.BASEL_PASSWORD,
                app.AIRPORT_IMPORTER_RETRY_SETTINGS,
                basel_settings.BASEL_REQUEST_TIMEOUT,
                basel_settings.BASEL_OPERATION_TIMEOUT,
                proxy_pool=proxy_pool.PARTNERS_PROXY_POOL,
            ),
            flight_number_parser=self._flight_number_parser,
            logger=logging.getLogger('basel_importer'),
        )

    def _get_importer(self, airport: str) -> AirportImporter:
        region_airports = {iata: partial(self.region_airports_updater, iata) for iata in region.REGION_AIRPORT_IATA}

        basel = {iata: partial(self.basel_updater, iata) for iata in basel_settings.BASEL_IATAS}

        return {
            'dme': self.dme_updater,
            'led': self.led_updater,
            'svo': self.svo_updater,
            'vko': self.vko_updater,
            **region_airports,
            **basel,
        }[airport]()

    def update(self, airport: str, *args, **kwargs):
        importer = self._get_importer(airport)

        try:
            status_data_pack = importer.collect_statuses(*args, **kwargs)
            if status_data_pack is not None:
                update_statuses(status_data_pack.statuses, airport)
                send_raw_data(status_data_pack, airport)
            else:
                self._logger.error('%s importer returned `None` status pack', airport)

        except Exception as e:
            self._logger.exception('%s importer finished with exception: %s', airport, e)

        return importer.statistics

    def collect_statuses(self, airport: str, *args, **kwargs) -> StatusDataPack:
        importer = self._get_importer(airport)

        return importer.collect_statuses(*args, **kwargs)
