from typing import Dict, Iterable, NamedTuple, Optional

from travel.proto.dicts.rasp.carrier_pb2 import TCarrier

from travel.avia.library.python.iata_correction import IATACorrector
from travel.avia.library.python.shared_dicts.cache.company_cache import CompanyCache

FlightNumber = NamedTuple(
    'FlightNumber',
    [
        ('company_code', str),
        ('company_id', int),
        ('number', str),
    ],
)


class FlightNumberParser(object):
    def __init__(self, companies: CompanyCache, iata_corrector: IATACorrector):
        self._companies = companies
        self._iata_corrector = iata_corrector

    def parse(self, flight_number: str) -> FlightNumber:
        return self.parse_list([flight_number])[flight_number]

    def parse_list(self, flight_numbers: Iterable[str]) -> Dict[str, FlightNumber]:
        split_numbers = [
            tuple(number.split(maxsplit=1)) if ' ' in number else ('', number) for number in flight_numbers
        ]
        company_ids = self._iata_corrector.flight_numbers_to_carriers(split_numbers)

        result = {}
        for split_number in split_numbers:
            flight_number = ' '.join(split_number)

            if company_ids[flight_number]:
                company = self._companies.get_company_by_id(company_ids[flight_number])
                company_code = self._companies.get_company_code_by_id(company_ids[flight_number])
            else:
                company = None
                company_code = None

            result[flight_number.lstrip()] = FlightNumber(
                company_code=split_number[0] or company_code,
                company_id=company_ids.get(flight_number),
                number=self._normalize_number(company_code, split_number[1], company),
            )

        return result

    @staticmethod
    def _normalize_number(company_code: str, number: str, company: Optional[TCarrier] = None) -> str:
        normalized = FlightNumberParser._remove_prefix(number.upper(), company_code)

        if company:
            for code in (company.Iata, company.Icao, company.SirenaId):
                normalized = FlightNumberParser._remove_prefix(normalized, code)

        return normalized.lstrip('0')

    @staticmethod
    def _remove_prefix(text: str, prefix: str) -> str:
        # TODO: replace with str.removeprefix after python 3.9
        # https://docs.python.org/3.9/library/stdtypes.html?highlight=removeprefix#str.removeprefix
        if text is None:
            return ''
        if prefix is None:
            return text
        if text.startswith(prefix):
            return text[len(prefix) :]
        return text
