import logging
import uuid
from abc import ABC
from dataclasses import dataclass
from datetime import datetime
from typing import List, Optional, Dict, Any, Union

import ujson as json

from travel.avia.flight_status_fetcher.services.status import IStatus

logger = logging.getLogger(__name__)


@dataclass
class StatusDataPack:
    message_id: str
    received_at: int
    statuses: List[Union[IStatus, Dict[str, Any]]]
    partner: str
    data: str
    error: Optional[str]

    def __post_init__(self):
        if not self.message_id:
            raise ValueError('Message id is not filled')

        if not self.received_at:
            raise ValueError('Received at is not filled')

        if not self.partner:
            raise ValueError('Partner value is not filled')

    def to_json_bytes(self):
        return json.dumps(
            {
                'message_id': self.message_id,
                'received_at': self.received_at,
                'statuses': [s.fields_dict for s in self.statuses],
                'partner': self.partner,
                'data': self.data,
                'error': self.error,
            }
        ).encode('utf-8')


class StatusDataCollector(ABC):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.partner: str = ''
        self.message_id: str = str(uuid.uuid4())
        self.received_at: int = int(datetime.now().timestamp())
        self.statuses = []
        self._raw_data: str = ''
        self.error = None

    def add_raw_data(self, data, encoding='utf-8'):
        if not data:
            return
        if isinstance(data, bytes):
            try:
                data = data.decode(encoding)
            except UnicodeDecodeError:
                try:
                    data = data.decode('latin-1')
                    logger.exception('Raw data decoding error. Fallback to latin-1')
                except Exception:
                    logger.exception('Raw data decoding error. Fallback latin-1 failed')
                    data = '<INVALID ENCODING>'
        self._raw_data += '%s' % data

    def status_data_pack(self):
        try:
            return StatusDataPack(
                message_id=self.message_id,
                received_at=self.received_at,
                statuses=self.statuses,
                partner=('airport:{}'.format(self.partner)).lower(),
                data=self._raw_data,
                error=self.error,
            )
        except Exception:
            logger.exception('Status data pack error')
            return None
