from typing import Dict, Any, Optional, Union, Tuple

from requests import Session
from requests.adapters import HTTPAdapter
from urllib3 import Retry


class Api:
    def __init__(
        self,
        base_url: str,
        retry_settings: Dict[str, Any],
        request_timeout: Optional[Union[float, Tuple[float, float]]] = 60,
    ):
        self._base_url = base_url
        session = Session()
        adapter = HTTPAdapter(
            max_retries=Retry(**retry_settings),
        )
        session.verify = False
        session.mount('https://', adapter)
        session.mount('http://', adapter)
        self._timeout = request_timeout
        self._session = session

    def _get_data(self, url: str, params=None, **kwargs):
        return self._fetch('get', url, params=params, **kwargs)

    def _post_data(self, url: str, params=None, **kwargs):
        return self._fetch('post', url, params=params, **kwargs)

    def _fetch(self, method: str, url: str, params=None, **kwargs):
        url = '{}{}'.format(self._base_url, url)
        method = {
            'get': self._session.get,
            'post': self._session.post,
            'head': self._session.head,
            'put': self._session.put,
        }.get(method.lower())

        if method is None:
            raise ValueError('Unsupported method: {}'.format(method))

        kwargs.setdefault('timeout', self._timeout)

        return method(url, params=params, **kwargs).json()
