"""Взаимодействие с avia-backend."""
import logging
import time
from typing import Any, Dict, Tuple, Optional, Union

from retrying import retry

from travel.avia.flight_status_fetcher import common
from travel.avia.flight_status_fetcher.services.api import Api

logger = logging.getLogger(__name__)


class AviaBackend(Api):
    AIRPORTS_URL = '/rest/airports'
    AIRLINES_INFO_URL = '/rest/airlines/airline_info'

    _instance = None

    @classmethod
    def get_instance(
        cls,
        base_url: str,
        retry_settings: Dict[str, Any],
        request_timeout: Optional[Union[float, Tuple[float, float]]] = None,
        timeout: float = 0,
    ):
        if cls._instance is None:
            cls._instance = cls(base_url, retry_settings, request_timeout)
        cls._instance._retry_settings = retry_settings
        cls._instance._refresh_reference(timeout)
        return cls._instance

    def __init__(
        self,
        base_url: str,
        retry_settings: Dict[str, Any],
        request_timeout: Optional[Union[float, Tuple[float, float]]],
    ):
        super().__init__(base_url, retry_settings, request_timeout)

        self.iata_by_aero_name = {}
        self.sirena_by_aero_name = {}

        self.iata_by_city_name = {}
        self.sirena_by_city_name = {}

        self.airline_id_by_sirena = {}  # type: Dict[str, int]
        self.airline_id_by_iata = {}  # type: Dict[str, int]

        self._last_refreshed_time = 0
        self._refresh_reference()

    def _refresh_reference(self, timeout: float = 0):
        current_time = time.time()
        time_since_last_refresh = current_time - self._last_refreshed_time
        logger.info(
            'Reference data was refreshed %s seconds ago. Current timeout is %s',
            time_since_last_refresh,
            timeout,
        )
        if time_since_last_refresh < timeout:
            logger.info('Using cached reference data.')
            return
        try:
            logger.info('Trying to refresh reference data.')
            (
                iata_by_aero_name,
                sirena_by_aero_name,
                iata_by_city_name,
                sirena_by_city_name,
            ) = self._get_airports_data()
            (
                airline_id_by_sirena,
                airline_id_by_iata,
            ) = self._get_airlines_data()
        except Exception:
            logger.exception('Failed to refresh reference data.')
        else:
            self.iata_by_aero_name = iata_by_aero_name
            self.sirena_by_aero_name = sirena_by_aero_name

            self.iata_by_city_name = iata_by_city_name
            self.sirena_by_city_name = sirena_by_city_name

            self.airline_id_by_sirena = airline_id_by_sirena
            self.airline_id_by_iata = airline_id_by_iata

            self._last_refreshed_time = current_time
            logger.info('Refreshed reference data successfully.')

    @retry(
        retry_on_result=lambda response: 'data' not in response,
        stop_max_attempt_number=5,
    )
    def _try_fetch(self, method, url, params=None, **kwargs):
        return super()._fetch(method, url, params, **kwargs)

    def _fetch(self, method: str, url: str, params=None, **kwargs):
        return self._try_fetch(method, url, params, **kwargs)['data']

    def _get_airports_data(self):
        iata_by_aero_name = {}
        sirena_by_aero_name = {}
        iata_by_city_name = {}
        sirena_by_city_name = {}

        for airport_data in self._get_data(self.AIRPORTS_URL):
            airport_name = airport_data.get('title')
            settlement_name = airport_data.get('settlement', {}).get('title')

            iata = airport_data.get('iataCode')
            sirena = airport_data.get('sirenaCode')

            if airport_name and iata:
                iata_by_aero_name[common.canonical_name_ru(airport_name)] = iata
            if airport_name and sirena:
                sirena_by_aero_name[common.canonical_name_ru(airport_name)] = sirena

            if settlement_name and iata:
                iata_by_city_name[common.canonical_name_ru(settlement_name)] = iata
            if settlement_name and sirena:
                sirena_by_city_name[common.canonical_name_ru(settlement_name)] = sirena

        return iata_by_aero_name, sirena_by_aero_name, iata_by_city_name, sirena_by_city_name

    def _get_airlines_data(self) -> Tuple[Dict[str, int], Dict[str, int]]:
        airline_id_by_sirena = {}
        airline_id_by_iata = {}

        for airline_data in self._get_data(self.AIRLINES_INFO_URL).values():
            if airline_data.get('hidden'):
                continue
            airline_id = int(airline_data.get('id'))
            sirena = airline_data.get('sirena')
            iata = airline_data.get('iata')
            if airline_id and sirena:
                airline_id_by_sirena[sirena] = airline_id
            if airline_id and iata:
                airline_id_by_iata[iata] = airline_id

        return airline_id_by_sirena, airline_id_by_iata
