import logging
import ujson as json
import uuid
from typing import Iterable


from travel.library.python.logbroker.writer import LogbrokerProducer, LogbrokerWriter
from travel.avia.flight_status_fetcher.library.raw_data import StatusDataPack
from travel.avia.flight_status_fetcher.services.status import IStatus, IStatusUpdater, UpdateStatusError

logger = logging.getLogger(__name__)


class LogbokerStatusProducer(IStatusUpdater, LogbrokerProducer):
    def update_status(self, statuses: Iterable[IStatus]):
        req_id = uuid.uuid4()
        try:
            logger.info(
                'Sending %s statuses to logbroker topic: %s , %s',
                str(len(list(statuses))),
                str(self.topic),
                str(req_id),
            )
            json_string = self.statuses_to_message(statuses)
            with LogbrokerWriter(self.token, self.topic, self.source_id, self.endpoint) as writer:
                write_result = writer.write(json_string)
            logger.info('Single message written with result: %s. %s', write_result, str(req_id))
            return write_result, req_id
        except Exception as err:
            raise UpdateStatusError(req_id) from err

    @staticmethod
    def statuses_to_message(statuses):
        data = [status.fields_dict for status in statuses]
        return json.dumps(data).encode('utf-8')


class RawDataLogbrokerProducer(LogbrokerProducer):
    def send_raw_data(self, raw_data: StatusDataPack):
        json_string = raw_data.to_json_bytes()
        try:
            with LogbrokerWriter(self.token, self.topic, self.source_id, self.endpoint) as writer:
                write_result = writer.write(json_string)
            logger.info('Raw data message written with result: %s. %s', write_result, str(raw_data.message_id))
            return write_result, raw_data.message_id
        except Exception as err:
            raise UpdateStatusError(raw_data.message_id) from err
