import abc
import datetime as dt
from typing import Any, Dict, Iterable, Optional, Union
from uuid import uuid4

from travel.avia.flight_status_fetcher.const import Direction

DT_FORMAT = '%Y-%m-%dT%H:%M:%S'
DATE_FORMAT = '%Y-%m-%d'


class IStatus(metaclass=abc.ABCMeta):
    @property
    @abc.abstractmethod
    def fields_dict(self) -> Dict[str, Any]:
        pass


# TODO(mikhailche): перепиши меня на прото-файлы после переноса в аркадию
# https://a.yandex-team.ru/arc/trunk/arcadia/travel/proto/avia/flight_status/flight_status.proto
# https://github.yandex-team.ru/avia/flight-storage/blob/release/src/github.yandex-team.ru/avia/flight-storage/storage/DTO/status.go#L15
class Status(IStatus):
    _known_fields = [
        'status_id',  # for logbroker pipeline. UUID4 string for debugging purposes
        'message_id',  # for logbroker pipeline. UUID4 string for debugging purposes
        'received_at',  # for logbroker pipeline
        'airport',
        'airline_id',
        'airline_code',  # for logbroker pipeline
        'flight_number',
        'flight_date',
        'direction',
        'time_actual',
        'time_scheduled',
        'status',
        'gate',
        'terminal',
        'check_in_desks',
        'baggage_carousels',
        'source',
        'diverted',
        'diverted_airport_iata',
        'diverted_airport_code',  # for logbroker pipeline. same as diverted_airport_iata, just a different name
        'route_point_from',  # new field, not in flight storage
        'route_point_to',  # new field, not in flight storage
    ]
    _known_fields_set = set(_known_fields)
    __slots__ = _known_fields

    def __init__(
        self,
        message_id: str,
        received_at: int,
        airport: str,
        airline_id: Optional[int],
        airline_code: Optional[str],
        flight_number: Optional[str],
        flight_date: Optional[dt.date],
        direction: Union[str, Optional[Direction]],
        time_actual: Optional[dt.datetime],
        time_scheduled: Optional[dt.datetime],
        status: str,
        gate: Optional[str],
        terminal: Optional[str],
        check_in_desks: Optional[str],
        baggage_carousels: Optional[str],
        source: str,
        diverted: bool = False,
        diverted_airport_iata: str = None,
        route_point_from: str = None,
        route_point_to: str = None,
    ):
        self.status_id = str(uuid4())
        self.message_id = message_id
        self.received_at = received_at

        self.airport = airport.upper()
        self.airline_id = airline_id
        self.airline_code = airline_code
        self.flight_number = flight_number

        if flight_date is not None:
            self.flight_date = flight_date.strftime(DATE_FORMAT)

        self.direction = direction

        self.time_actual = time_actual.strftime(DT_FORMAT) if time_actual else None
        self.time_scheduled = time_scheduled.strftime(DT_FORMAT) if time_scheduled else None

        self.status = status
        self.diverted = diverted
        self.diverted_airport_iata = diverted_airport_iata
        self.diverted_airport_code = diverted_airport_iata
        self.gate = gate
        self.terminal = terminal
        self.check_in_desks = check_in_desks
        self.baggage_carousels = baggage_carousels
        self.source = source
        self.route_point_from = route_point_from
        self.route_point_to = route_point_to

    @property
    def fields_dict(self) -> Dict[str, Any]:
        return {field_name: getattr(self, field_name, None) for field_name in self._known_fields}


class UpdateStatusError(Exception):
    def __init__(self, req_id, *args):
        self.req_id = req_id
        super(UpdateStatusError, self).__init__(req_id, *args)


class IStatusUpdater:
    @abc.abstractmethod
    def update_status(self, statuses: Iterable[IStatus]):
        pass
