"""Retrieving data from Basel Aero"""
import logging
from datetime import datetime, timedelta
from typing import Dict, Iterable, Optional

from parsel import Selector
from zeep import Client

from travel.avia.flight_status_fetcher import const
from travel.avia.flight_status_fetcher.const import FlightStatus, Direction
from travel.avia.flight_status_fetcher.library.airport_importer import AirportImporter
from travel.avia.flight_status_fetcher.library.flight_number_parser import FlightNumber, FlightNumberParser
from travel.avia.flight_status_fetcher.library.raw_data import StatusDataPack
from travel.avia.flight_status_fetcher.services.status import Status


class BaselImporter(AirportImporter):
    DATETIME_FORMAT = '%d.%m.%Y %H:%M:%S'
    IGNORED_AIRPORTS = {'На Точку'}
    IGNORED_AIRPORT_IATAS = {'301'}

    def __init__(
        self,
        iata: str,
        wsdl_client: Client,
        flight_number_parser: FlightNumberParser,
        logger: logging.Logger,
    ):
        super().__init__(logger, flight_number_parser)
        self.iata = iata
        self.client = wsdl_client
        self.status_data_collector.partner = iata
        self.checkin_normalizer = BaselCheckinDesksNormalizer(logger)

    def _get_route_points(self, flight, iata: str, direction: Direction):
        if not direction:
            self.logger.warning('Could not determine route points because direction is empty', stack_info=True)
            return None, None
        airport = flight.attrib.get('AirportIATA')
        if direction == Direction.ARRIVAL:
            return airport, iata.upper()
        return iata.upper(), airport

    @staticmethod
    def _extract_flight_numbers(flight, **kwargs):
        airline_code = flight.attrib.get('AirlineCode')
        flight_number = flight.attrib.get('FlightNumber')

        # Если Базел не прислал код авиакомпании, отдаем право определения в shared-flights
        if airline_code == '0':
            airline_code = ''

        return ['{} {}'.format(airline_code, flight_number).lstrip()]

    def _valid(self, flight) -> bool:
        return (
            flight.attrib.get('DepartureArrivalAirport') not in self.IGNORED_AIRPORTS
            and flight.attrib.get('AirportIATA') not in self.IGNORED_AIRPORT_IATAS
        )

    def build_statuses(self, xml: str, iata: str) -> Iterable[Status]:
        selector = Selector(text=xml, type='xml')
        elements = []
        for element in selector.xpath('/Данные/Элемент'):
            if not self._valid(element):
                continue
            elements.append(element)

        flight_numbers = self._parse_flight_numbers(elements)

        message_id = self.status_data_collector.message_id
        received_at = int(datetime.now().timestamp())
        self.status_data_collector.received_at = received_at
        for element in elements:
            try:
                for status in self._build_status(element, flight_numbers, iata, received_at, message_id):
                    yield status
                    self._last_run_statistics['success'] += 1
            except Exception:
                self.logger.error('Could not transform flight %s into status:', element, exc_info=True)
                self._last_run_statistics['failure'] += 1

    @staticmethod
    def parse_direction(direction: str):
        if direction == 'Прилетной':
            return Direction.ARRIVAL
        if direction == 'Вылетной':
            return Direction.DEPARTURE
        if direction == 'Возвратный':
            return None
        raise ValueError('Unknown direction: "{}"'.format(direction))

    def _build_status(
        self,
        element,
        flight_numbers: Dict[str, FlightNumber],
        iata: str,
        received_at: int,
        message_id: str,
    ):
        direction = self.parse_direction(element.attrib.get('Sign'))
        if not direction:
            return
        route_point_from, route_point_to = self._get_route_points(element, iata, direction)

        for flight_number in self._extract_flight_numbers(element):
            if not flight_numbers[flight_number].company_code:
                continue

            yield Status(
                message_id=message_id,
                airport=iata.upper(),
                airline_id=flight_numbers[flight_number].company_id,
                airline_code=flight_numbers[flight_number].company_code,
                flight_number=flight_numbers[flight_number].number,
                flight_date=self._parse_datetime(element.attrib.get('TimePlan')).date(),
                direction=direction.value if direction else None,
                time_actual=self._get_actual_time(
                    self._parse_datetime(element.attrib.get('ETA'), allow_none=True),
                    self._parse_datetime(element.attrib.get('TimeFact'), allow_none=True),
                ),
                time_scheduled=self._parse_datetime(element.attrib.get('TimePlan'), allow_none=True),
                status=self.parse_status(element.attrib.get('Status')).value,
                gate=element.attrib.get('GateNumber'),
                terminal=element.attrib.get('Terminal'),
                check_in_desks=self.checkin_normalizer.normalize_checkin_desks(element.attrib.get('ReceptionDesk')),
                baggage_carousels=element.attrib.get('BaggageNumber'),
                source=const.STATUS_SOURCE_AIRPORT,
                route_point_from=route_point_from,
                route_point_to=route_point_to,
                received_at=received_at,
            )

    def parse_status(self, status: str) -> FlightStatus:
        if not status:
            return FlightStatus.STATUS_NO_DATA_VALUE
        status_map = {
            'DROP OFF': FlightStatus.STATUS_WAIT,
            'В ПУТИ': FlightStatus.STATUS_WAIT,
            'ВЫДАЧА БАГАЖА': FlightStatus.STATUS_ARRIVED,
            'ВЫДАЧА ЗАВЕРШЕНА': FlightStatus.STATUS_ARRIVED,
            'ВЫЛЕТЕЛ': FlightStatus.STATUS_DEPARTED,
            'ЗАДЕРЖАН': FlightStatus.STATUS_DELAY,
            'ОТМЕНЕН': FlightStatus.STATUS_CANCELLED,
            'ОТПРАВЛЯЕТСЯ': FlightStatus.STATUS_BOARDING_FINISHED,
            'ПО РАСПИСАНИЮ': FlightStatus.STATUS_WAIT,
            'ПОСАДКА ОКОНЧЕНА': FlightStatus.STATUS_BOARDING_FINISHED,
            'ПОСАДКА': FlightStatus.STATUS_BOARDING,
            'ПРИБЫЛ': FlightStatus.STATUS_ARRIVED,
            'РЕГИСТРАЦИЯ ЗАКОНЧЕНА': FlightStatus.STATUS_REGISTRATION_FINISHED,
            'РЕГИСТРАЦИЯ': FlightStatus.STATUS_REGISTRATION,
            'РЕГИСТРАЦИЯ/ПОСАДКА': FlightStatus.STATUS_BOARDING,
            'СОВЕРШИЛ ПОСАДКУ': FlightStatus.STATUS_ARRIVED,
            'СОВМЕЩЕН': FlightStatus.STATUS_CANCELLED,
        }
        if status not in status_map:
            self.logger.warning('Unknown status: %s', status, stack_info=True)
            return FlightStatus.STATUS_UNKNOWN_VALUE
        return status_map[status]

    def collect_statuses(self, date_from: datetime = None, date_to: datetime = None) -> StatusDataPack:
        self.logger.info('Basel: update started')
        self._clear_statistics()
        if not date_from:
            date_from = (datetime.now() - timedelta(days=2)).date()
        if not date_to:
            date_to = (datetime.now() + timedelta(days=30)).date()
        try:
            xml = self.client.service.send(date_from, date_to, self.iata)
            self.status_data_collector.add_raw_data(xml)
            self._last_run_statistics['datasize'] = len(xml)
            self.status_data_collector.statuses = list(self.build_statuses(xml, self.iata))
        except Exception as e:
            self.status_data_collector.error = 'Basel: cannot download status data: {}'.format(e)
            self.logger.exception('Basel: cannot download status data')
        finally:
            self.logger.info('Basel: import finished')
            return self.status_data_collector.status_data_pack()


class BaselCheckinDesksNormalizer:
    def __init__(self, logger=None):
        self.logger = logger or logging.getLogger(__name__)

    def normalize_checkin_desks(self, desks_str: Optional[str]) -> Optional[str]:
        """
        Converts comma-separated unordered list of desks to a nice range-grouped representation

        8,8,6,7 -> 6-8
        4,6,7,8 -> 4, 6-8
        :param str desks_str: comma-separated unordered list of desks like '8,8,6,7,9-12'
        :return: range-grouped representation like '4, 6-8'
        """
        if not desks_str:
            return desks_str
        desks_str = desks_str.strip()
        if not desks_str:
            return ''
        splitters = '''−‐‑-ー一_'''
        if any(splitter in desks_str for splitter in splitters):
            # not supposed to happen according to previous data
            return desks_str

        desks = {desk.strip() for desk in desks_str.split(',')}

        try:
            desks_int = {int(desk) for desk in desks if desk}
        except ValueError:
            self.logger.exception('Int expected')
            return desks_str

        ranges = []
        last_desk: Optional[int] = None

        for desk in sorted(desks_int):
            if last_desk is None or desk - 1 > last_desk:
                ranges.append((desk, desk))
            ranges[-1] = ranges[-1][0], desk
            last_desk = desk

        return ', '.join(
            (str(range_from) if range_from == range_to else str(range_from) + '-' + str(range_to))
            for range_from, range_to in ranges
        )
