""" Flight status importer from DME airport """
import logging
import re
from datetime import datetime, timedelta
from urllib.parse import urlencode

from lxml import etree
from typing import Dict, Optional

from travel.avia.library.python.proxy_pool.proxy_pool import ProxyPool

from travel.avia.flight_status_fetcher.library.airport_importer import AirportHTTPImporter
from travel.avia.flight_status_fetcher.library.flight_number_parser import FlightNumber, FlightNumberParser
from travel.avia.flight_status_fetcher.library.raw_data import StatusDataPack
from travel.avia.flight_status_fetcher.services.status import Status
from travel.avia.flight_status_fetcher.utils.safe_xml_parser import safe_xml_fromstring


def _get_from_first_existing_key(dictionary, *keys):
    for key in keys:
        value = dictionary.get(key)
        if value:
            return value


class FiltersFlights:
    @staticmethod
    def by_nat_equal_cgo(flight):
        return flight.get('NAT') != 'CGO'

    @staticmethod
    def by_fl_num_pub(flight):
        flight_number = flight.get('FL_NUM_PUB')
        return flight_number is not None and flight_number != ''


class DMEImporter(AirportHTTPImporter):
    AIRPORT = 'DME'
    DATETIME_FORMAT = '%Y-%m-%dT%H:%M:%S'
    RE_LAST_URL = re.compile(r'<a\s+href="([\d_\-]+/)">.*\n</pre>')
    RE_SINGLE_XML = re.compile(r'<a\s+href="(.+\.xml)">')

    _default_filters = [
        FiltersFlights.by_nat_equal_cgo,
        FiltersFlights.by_fl_num_pub,
    ]

    def __init__(
        self,
        data_url,
        retry_settings,
        logger: logging.Logger,
        flight_number_parser: FlightNumberParser,
        request_timeout,
        proxy_pool: Optional[ProxyPool] = None,
    ):
        super().__init__(retry_settings, logger, flight_number_parser, proxy_pool)
        self._data_url = data_url
        self._request_timeout = request_timeout
        self.status_data_collector.partner = self.AIRPORT

    @staticmethod
    def _get_gate(flight, direction):
        return None if direction == 'arrival' else flight.get('GATE')

    @staticmethod
    def _get_status(flight):
        return 'cancelled' if flight.get('CANCELLED') == '1' else 'delay' if flight.get('DELAYED') == '1' else 'unknown'

    @staticmethod
    def _get_company_code_with_flight_number(flight):
        return flight.get('FL_NUM_PUB').split()[:2]

    def _build_status(self, flight, direction, flight_numbers: Dict[str, FlightNumber], message_id, received_at):
        scheduled_time = self._parse_datetime(flight.get('TIM_P'))
        expected_time = self._parse_datetime(flight.get('TIM_F'), allow_none=True)
        actual_time = self._parse_datetime(_get_from_first_existing_key(flight, 'TIM_L', 'TIM_D'), allow_none=True)

        route_point_from, route_point_to = self._route_points(flight.get('ROUTE_R'), direction)

        for flight_number in self._extract_flight_numbers(flight):
            if not flight_numbers[flight_number].company_code:
                continue

            yield Status(
                message_id=message_id,
                received_at=received_at,
                airline_id=flight_numbers[flight_number].company_id,
                airline_code=flight_numbers[flight_number].company_code,
                flight_number=flight_numbers[flight_number].number,
                direction=direction,
                airport=self.AIRPORT,
                flight_date=scheduled_time.date(),
                status=self._get_status(flight),
                terminal=None,
                gate=self._get_gate(flight, direction),
                time_actual=self._get_actual_time(expected_time, actual_time),
                time_scheduled=scheduled_time,
                check_in_desks=flight.get('REGPOINT'),
                source=self.SOURCE,
                baggage_carousels=None,
                route_point_from=route_point_from,
                route_point_to=route_point_to,
            )

    def _get_latest_data_url(self):
        msk_today = self._utc_to_msk(datetime.utcnow())
        date_from = (msk_today - timedelta(1)).strftime('%Y-%m-%d')
        date_to = (msk_today + timedelta(3)).strftime('%Y-%m-%d')
        return '{}?{}'.format(self._data_url, urlencode({'from': date_from, 'to': date_to}))

    def _get_all_company_codes_with_flight_numbers(self, flights):
        return [self._get_company_code_with_flight_number(flight) for flight in flights]

    @classmethod
    def _filter_flights(cls, flights, filters=None):
        filters = filters or cls._default_filters
        for func in filters:
            flights = filter(func, flights)

        return flights

    @staticmethod
    def _extract_flight_numbers(flight, **kwargs):
        return [' '.join(flight.get('FL_NUM_PUB').split()[:2])]

    def _build_statuses(self, latest_data=None):
        if not latest_data:
            latest_data_url = self._get_latest_data_url()
            self.logger.info('Latest file found: %s', latest_data_url)

            latest_data = self._curl(
                latest_data_url,
                timeout=self._request_timeout,
            )
        self.status_data_collector.add_raw_data(latest_data)
        try:
            tree = safe_xml_fromstring(latest_data)
        except etree.XMLSyntaxError as e:
            self.logger.error('Bad XML from dmd: %s', e)
            raise

        message_id = self.status_data_collector.message_id
        received_at = int(datetime.now().timestamp())
        self.status_data_collector.received_at = received_at

        arrival_flights = tree.xpath('//DMDTablo/DMDTabloArrive/DMDTabloArriveFlight')
        departure_flights = tree.xpath('//DMDTablo/DMDTabloDeparture/DMDTabloDepartureFlight')

        arrival_flights = list(self._filter_flights(arrival_flights))
        departure_flights = list(self._filter_flights(departure_flights))

        arrival_flight_numbers = self._parse_flight_numbers(arrival_flights)
        departure_flight_numbers = self._parse_flight_numbers(departure_flights)

        for flight in arrival_flights:
            yield from self._build_status(
                flight,
                'arrival',
                arrival_flight_numbers,
                message_id=message_id,
                received_at=received_at,
            )
            self._last_run_statistics['success'] += 1

        for flight in departure_flights:
            yield from self._build_status(
                flight,
                'departure',
                departure_flight_numbers,
                message_id=message_id,
                received_at=received_at,
            )
            self._last_run_statistics['success'] += 1

    def collect_statuses(self, latest_data=None) -> StatusDataPack:
        self.logger.info('DME update started')
        self._clear_statistics()
        try:
            self.status_data_collector.statuses = list(self._build_statuses(latest_data))
        except Exception as e:
            self.status_data_collector.error = 'DME update error: {}'.format(e)
            self.logger.exception('DME update error')
        finally:
            self.logger.info('DME update finished')
            return self.status_data_collector.status_data_pack()

    def _route_points(self, route_string: str, direction: str):
        points = route_string.split('->')
        dme_index = None
        for i, point in enumerate(points):
            if 'домодедово' in point.lower():
                dme_index = i
                break
        default_result = (None, self.AIRPORT) if direction == 'arrival' else (self.AIRPORT, None)
        if dme_index is None:
            return default_result
        if direction == 'arrival':
            if dme_index - 1 < 0:
                return default_result
            return points[dme_index - 1], self.AIRPORT
        elif direction == 'departure':
            if dme_index + 1 >= len(points):
                return default_result
            return self.AIRPORT, points[dme_index + 1]
        else:
            self.logger.error('Unknown direction: %s', direction)
