""" Flight status importer from VKO airport """
import logging
import re
import time
from copy import copy
from datetime import datetime
from typing import Dict

from lxml import etree

from travel.avia.flight_status_fetcher.ftp_wrapper import FTPWrapper
from travel.avia.flight_status_fetcher.library.airport_importer import AirportImporter
from travel.avia.flight_status_fetcher.library.flight_number_parser import FlightNumber, FlightNumberParser
from travel.avia.flight_status_fetcher.library.raw_data import StatusDataPack
from travel.avia.flight_status_fetcher.services.status import Status
from travel.avia.flight_status_fetcher.utils.safe_xml_parser import safe_xml_fromstring
from travel.avia.flight_status_fetcher.utils.xml import xml_to_dict


class LEDImporter(AirportImporter):
    AIRPORT = 'LED'
    RE_LAST_URL = re.compile(r'<a href="([\d_\-]+/)">.*\n</pre>')
    DEPARTURE = 'departure'
    ARRIVAL = 'arrival'
    DATETIME_FORMAT = '%Y-%m-%dT%H:%M:%S.%f'

    def __init__(
        self,
        ftp_host,
        ftp_user,
        ftp_password,
        arrival_data_filename,
        departure_data_filename,
        retry_settings,
        flight_number_parser: FlightNumberParser,
        logger: logging.Logger,
        timeout,
        proxy_pool=None,
    ):
        super().__init__(logger, flight_number_parser)
        self._ftp_host = ftp_host
        self._ftp_user = ftp_user
        self._ftp_password = ftp_password
        self._arrival_data_filename = arrival_data_filename
        self._departure_data_filename = departure_data_filename
        self._timeout = timeout
        self._retry_settings = retry_settings
        self._proxy_pool = proxy_pool
        self.status_data_collector.partner = self.AIRPORT

        self._clear_statistics()

    @staticmethod
    def _get_gate(data_dict, direction):
        return None if direction == 'arrival' else data_dict.get('OD_GATES')

    @staticmethod
    def _parse_status(status):
        parsed_status = {'Canceled': 'cancelled', 'Departed': 'departed', 'Arrived': 'arrived', 'Delayed': 'delay'}.get(
            status
        )
        if not parsed_status and status:
            parsed_status = status
        return parsed_status

    def _extract_scheduled_expected_and_actual_time(self, data_dict, direction):
        if direction == self.DEPARTURE:
            scheduled_time = data_dict.get('OD_STD')
            expected_time = data_dict.get('OD_ETD')
            actual_time = data_dict.get('OD_ATD')
        else:
            scheduled_time = data_dict.get('OA_STA')
            expected_time = data_dict.get('OA_ETA')
            actual_time = data_dict.get('OA_ATA')
        return (
            self._parse_datetime(scheduled_time),
            self._parse_datetime(expected_time, allow_none=True),
            self._parse_datetime(actual_time, allow_none=True),
        )

    def _make_status_without_flight_number(self, data_dict, direction, message_id, received_at):
        scheduled_time, expected_time, actual_time = self._extract_scheduled_expected_and_actual_time(
            data_dict, direction
        )
        status = data_dict.get('OD_STATUS_EN') if direction == self.DEPARTURE else data_dict.get('OA_STATUS_EN')
        route_point_from = None
        route_point_to = None
        if direction == self.ARRIVAL:
            route_point_from = data_dict.get('OA_RAP_CODE_PREVIOUS') or data_dict.get('OA_RAP_CODE_ORIGIN')
            route_point_to = self.AIRPORT
        if direction == self.DEPARTURE:
            route_point_from = self.AIRPORT
            route_point_to = data_dict.get('OD_RAP_CODE_NEXT') or data_dict.get('OD_RAP_CODE_DESTINATION')
        return Status(
            message_id=message_id,
            received_at=received_at,
            airline_id=None,
            airline_code=None,
            flight_number=None,
            baggage_carousels=None,
            direction=direction or None,
            airport=self.AIRPORT,
            flight_date=scheduled_time.date() if scheduled_time else None,
            status=self._parse_status(status) or 'no-data',
            terminal=None,
            gate=self._get_gate(data_dict, direction) or None,
            time_actual=self._get_actual_time(expected_time, actual_time) or None,
            time_scheduled=scheduled_time or None,
            check_in_desks=data_dict.get('OD_COUNTERS', '').replace('A', '') or None,
            source=self.SOURCE,
            route_point_from=route_point_from,
            route_point_to=route_point_to,
        )

    @staticmethod
    def _extract_flight_numbers(data_dict, **kwargs):
        flight_numbers = set()

        def extract_flight_number(index):
            add_end = '' if index == 0 else '_K{}'.format(index)
            return data_dict.get('O{}_FLIGHT_NUMBER{}'.format(direction_char, add_end))

        direction_char = kwargs['direction'][0].upper()
        i = 0
        while True:
            flight_number = extract_flight_number(i)
            if not flight_number:
                break
            if ' ' in flight_number:
                parts = flight_number.split()
                flight_number = ' '.join(parts[:2])
            flight_numbers.add(flight_number)
            i += 1
        return flight_numbers

    def _process_flight(
        self,
        flight_dict,
        direction,
        flight_numbers: Dict[str, FlightNumber],
        message_id,
        received_at,
    ):
        try:
            status = self._make_status_without_flight_number(flight_dict, direction, message_id, received_at)

            for flight_number in self._extract_flight_numbers(flight_dict, direction=direction):
                if not flight_numbers[flight_number].company_code:
                    continue

                status_copy = copy(status)
                status_copy.airline_id = flight_numbers[flight_number].company_id
                status_copy.airline_code = flight_numbers[flight_number].company_code
                status_copy.flight_number = flight_numbers[flight_number].number

                yield status_copy

        except Exception:
            self.logger.exception(
                'Could not parse %s flight %s into json',
                direction,
                flight_dict,
            )

    def _get_flights_from_xml(self, xml, direction):
        try:
            tree = safe_xml_fromstring(xml)
            return map(xml_to_dict, tree.xpath('//Flights/row'))
        except etree.XMLSyntaxError as e:
            self.logger.error('Bad XML from LED for direction %s.\nException: %s.\nContent: %s', direction, e, xml)
            raise RuntimeError('Bad XML from LED for direction {}. Content: {}'.format(direction, xml))

    def _fill_modification_time_stats(self, ftp):
        now = time.time()
        to_process = (('arrival', self._arrival_data_filename), ('departure', self._departure_data_filename))
        for direction, filename in to_process:
            try:
                modification_time = ftp.get_modification_time(filename)
                self._last_run_statistics['{}_age'.format(direction)] = now - modification_time.timestamp()
            except Exception as e:
                self.logger.exception('"%s" during getting modification time for %s', e, filename)

    def _build_flights(self, ftp):
        self._fill_modification_time_stats(ftp)
        arrival_xml = ftp.retrieve_string(self._arrival_data_filename)
        self.status_data_collector.add_raw_data('\n===ARRIVAL XML===\n\n\n')
        self.status_data_collector.add_raw_data(arrival_xml)
        self.logger.info('Got arrival xml')
        departure_xml = ftp.retrieve_string(self._departure_data_filename)
        self.status_data_collector.add_raw_data('\n===DEPARTURE XML===\n\n\n')
        self.status_data_collector.add_raw_data(departure_xml)
        self.logger.info('Got departure xml')
        yield from self._build_flights_from_strings(arrival_xml, departure_xml)

    def _build_flights_from_strings(self, arrival_xml, departure_xml):
        statuses = set()
        arrival_flights = list(self._get_flights_from_xml(arrival_xml, self.ARRIVAL))
        departure_flights = list(self._get_flights_from_xml(departure_xml, self.DEPARTURE))

        arrival_flight_numbers = self._parse_flight_numbers(arrival_flights, direction=self.ARRIVAL)
        departure_flight_numbers = self._parse_flight_numbers(departure_flights, direction=self.DEPARTURE)

        message_id = self.status_data_collector.message_id
        received_at = int(datetime.now().timestamp())
        self.status_data_collector.received_at = received_at

        for flight in arrival_flights:
            statuses.update(
                self._process_flight(
                    flight,
                    self.ARRIVAL,
                    arrival_flight_numbers,
                    message_id,
                    received_at,
                )
            )

        for flight in departure_flights:
            statuses.update(
                self._process_flight(
                    flight,
                    self.DEPARTURE,
                    departure_flight_numbers,
                    message_id,
                    received_at,
                )
            )

        for status in statuses:
            yield status
            self._last_run_statistics['success'] += 1

    def _build_statuses(self):
        ftp = FTPWrapper(
            host=self._ftp_host,
            user=self._ftp_user,
            passwd=self._ftp_password,
            timeout=self._timeout,
            retry_kwargs=self._retry_settings,
            proxy_pool=self._proxy_pool,
        )
        yield from self._build_flights(ftp)

    def collect_statuses(self) -> StatusDataPack:
        self.logger.info('LED update started')
        self._clear_statistics()
        try:
            self.status_data_collector.statuses = list(self._build_statuses())
        except Exception as e:
            self.status_data_collector.error = 'LED updated error: {}'.format(e)
            self.logger.exception('LED updated error')
        finally:
            self.logger.info('LED update finished')
            return self.status_data_collector.status_data_pack()
