""" Flight status importer from Region Airports"""
import logging
from collections import namedtuple
from datetime import datetime
from typing import Dict, List, Optional

from lxml import etree

from travel.avia.library.python.proxy_pool.proxy_pool import ProxyPool

from travel.avia.flight_status_fetcher.library.airport_importer import AirportHTTPImporter
from travel.avia.flight_status_fetcher.library.flight_number_parser import FlightNumber, FlightNumberParser
from travel.avia.flight_status_fetcher.library.raw_data import StatusDataPack
from travel.avia.flight_status_fetcher.services.status import Status
from travel.avia.flight_status_fetcher.utils.safe_xml_parser import safe_xml_fromstring

airport_code_groups = {
    'svx': {'ЕКБ', 'КОЛЬЦОВО', 'ЕКАТЕРИНБУРГ'},
    'goj': {'НЖС', 'СТРИГИНО', 'НИЖНИЙ НОВГОРОД'},
    'kuf': {'СКЧ', 'КУРУМОЧ', 'САМАРА'},
    'rov': {'РОВ', 'ПЛАТОВ', 'РОСТОВ-НА-ДОНУ'},
    'pkc': {'ПРЛ', 'ЕЛИЗОВО', 'ПЕТРОПАВЛОВСК-КАМЧАТСКИЙ'},
    'nux': {'НУР', 'НОВЫЙ УРЕНГОЙ'},
    'gsv': {'ГСВ', 'ГАГАРИН', 'САРАТОВ'},
}

airport_code_groups = {k.upper(): {v.upper() for v in vv} for k, vv in airport_code_groups.items()}


class RegionAirportImporter(AirportHTTPImporter):
    DATETIME_FORMAT = '%d.%m.%Y %H:%M'
    FlightTimes = namedtuple('FlightTimes', ['scheduled_time', 'expected_time', 'actual_time', 'flight_date'])
    IGNORED_VIDDS = {'Бизнесавиация', 'Бизнессавиация', 'Грузовой', 'Облет Аэронавигац. обор.'}
    IGNORED_NAME_AKS = {'Без Авиакомпании (Межд)'}

    def __init__(
        self,
        iata: str,
        retry_settings,
        logger: logging.Logger,
        flight_number_parser: FlightNumberParser,
        request_timeout,
        iata_to_url,
        proxy_pool: Optional[ProxyPool] = None,
    ):
        super().__init__(retry_settings, logger, flight_number_parser, proxy_pool)
        self.iata = iata
        self._request_timeout = request_timeout
        self._iata_to_url = iata_to_url
        self.status_data_collector.partner = iata

    @staticmethod
    def _parse_direction(direction):
        return {'В': 'departure', 'П': 'arrival'}[direction]

    @staticmethod
    def _get_status(flight):
        if flight.attrib.get('OTM') == 'О':
            return 'cancelled'
        if flight.attrib.get('PRIB') == 'OFF':
            return 'delay'
        if flight.attrib.get('TIP') == 'П':
            return 'arrived'
        if flight.attrib.get('TIP') == 'В':
            return 'departed'
        return 'no-data'

    @staticmethod
    def _get_route_points(flight, idx, direction):
        if direction == 'departure':
            first_idx = idx
            second_idx = idx + 1
        elif direction == 'arrival':
            first_idx = idx - 1
            second_idx = idx
        else:
            raise ValueError('Unknown direction %s', direction)

        points = RegionAirportImporter._get_all_route_points(flight)
        return points[first_idx], points[second_idx]

    @staticmethod
    def _get_all_route_points(flight) -> List[str]:
        return [f.attrib['AP'].rsplit('/')[-1] for f in flight]

    def _get_times(self, flight, idx):
        direction = self._parse_direction(flight.attrib['TIP'])
        if direction == 'departure':
            route = flight[idx]
            time_key_suffix = 'D'
        else:
            route = flight[idx]
            time_key_suffix = 'A'

        scheduled_time = self._parse_datetime(route.attrib.get('DP_{}'.format(time_key_suffix)), allow_none=True)
        actual_time = self._parse_datetime(route.attrib.get('DF_{}'.format(time_key_suffix)), allow_none=True)
        expected_time = self._parse_datetime(route.attrib.get('DR_{}'.format(time_key_suffix)), allow_none=True)
        flight_date = scheduled_time.date() if scheduled_time else None
        return self.FlightTimes(scheduled_time, expected_time, actual_time, flight_date)

    @staticmethod
    def _extract_flight_numbers(flight, **kwargs):
        code = flight.attrib.get('IATA_AK')
        if not code:
            code = flight.attrib.get('ZRT_AK')
        if not code:
            code = flight.attrib.get('ICAO_AK')
        return ['{} {}'.format(code, flight.attrib.get('NR')).strip()]

    def _prepare_check_in_desks(self, flight) -> Optional[str]:
        raw_economy_desks = flight.attrib.get('DESK_E')
        raw_business_desks = flight.attrib.get('DESK_B')
        raw_check_in_desks = []
        for deck in raw_economy_desks.split(',') + raw_business_desks.split(','):
            if deck:
                raw_check_in_desks.append(deck)
        return self._join_check_in_desks(raw_check_in_desks)

    def _make_statuses(self, flight, flight_numbers: Dict[str, FlightNumber], message_id, received_at, iata: str):
        try:
            airport_index = self._get_airport_index(flight, iata)
        except AirportIndexError:
            self.logger.exception('Could not get airport (%s) index for flight %s', iata, flight)
            return
        self.logger.debug('Found airport %s index = %d in flight %s', iata, airport_index, flight)
        times = self._get_times(flight, airport_index)
        direction = self._parse_direction(flight.attrib.get('TIP')) or None
        route_point_from, route_point_to = self._get_route_points(flight, airport_index, direction)

        for flight_number in self._extract_flight_numbers(flight):
            if not flight_numbers[flight_number].company_code:
                continue

            yield Status(
                message_id=message_id,
                received_at=received_at,
                airport=iata.upper(),
                airline_id=flight_numbers[flight_number].company_id,
                airline_code=flight_numbers[flight_number].company_code,
                flight_number=flight_numbers[flight_number].number,
                flight_date=times.flight_date,
                direction=direction,
                status=self._get_status(flight),
                gate=flight.attrib.get('POS') or None,
                terminal=flight.attrib.get('TERM') or None,
                time_actual=self._get_actual_time(
                    times.expected_time,
                    times.actual_time,
                ),
                time_scheduled=times.scheduled_time,
                check_in_desks=self._prepare_check_in_desks(flight) or None,
                baggage_carousels=flight.attrib.get('BAG') or None,
                source=self.SOURCE,
                route_point_from=route_point_from,
                route_point_to=route_point_to,
            )

    def _download_xml(self, iata):
        return self._curl(
            self._iata_to_url[iata],
            timeout=self._request_timeout,
        )

    def _filter(self, flight) -> bool:
        return (
            flight.attrib.get('VIDD') not in self.IGNORED_VIDDS
            and flight.attrib.get('NAME_AK') not in self.IGNORED_NAME_AKS
        )

    def build_statuses(self, xml, iata):
        try:
            tree = safe_xml_fromstring(xml)
        except etree.XMLSyntaxError as e:
            self.logger.error('Bad XML from %s: %s', iata, e)
            raise

        flights = list(filter(self._filter, tree.xpath('//DATA/FLIGHT')))

        flight_numbers = self._parse_flight_numbers(flights)

        message_id = self.status_data_collector.message_id
        received_at = int(datetime.now().timestamp())
        self.status_data_collector.received_at = received_at
        for flight in flights:
            try:
                for status in self._make_statuses(flight, flight_numbers, message_id, received_at, iata):
                    yield status
                    self._last_run_statistics['success'] += 1
            except Exception:
                self.logger.exception(
                    'Could not transform flight %s into status',
                    flight.attrib,
                )
                self._last_run_statistics['failure'] += 1

    def collect_statuses(self) -> StatusDataPack:
        self.logger.info('Region airport {} update started'.format(self.iata))
        self._clear_statistics()
        try:
            xml = self._download_xml(self.iata)
            self.status_data_collector.add_raw_data(xml)
            self.status_data_collector.statuses = list(self.build_statuses(xml, self.iata))
        except Exception as e:
            self.status_data_collector.error = 'Region airport {} update error: {}'.format(self.iata, e)
            self.logger.exception('Region airport {} update error'.format(self.iata))
        finally:
            self.logger.info('Region airport {} update finished'.format(self.iata))
            return self.status_data_collector.status_data_pack()

    def _get_airport_index(self, flight, iata) -> int:
        iata = iata.upper()
        route_points = self._get_all_route_points(flight)
        try:
            possible_codes = airport_code_groups[iata]
        except KeyError as e:
            raise AirportIndexBadConfiguration(iata) from e
        for i, route_point in enumerate(route_points):
            if route_point.upper() in possible_codes:
                return i
        raise AirportIndexNotFound(iata)


class AirportIndexError(Exception):
    pass


class AirportIndexBadConfiguration(AirportIndexError):
    pass


class AirportIndexNotFound(AirportIndexError):
    pass
