"""Получение данных от Аэропортов Севера."""
import datetime as dt
import logging
import re
from datetime import datetime
from typing import Iterable, Optional, Tuple

from travel.avia.flight_status_fetcher import common
from travel.avia.flight_status_fetcher import const
from travel.avia.flight_status_fetcher.library.raw_data import StatusDataPack, StatusDataCollector
from travel.avia.flight_status_fetcher.library.statistics import ObjectWithStatistics
from travel.avia.flight_status_fetcher.services.avia_backend import AviaBackend
from travel.avia.flight_status_fetcher.services.status import Status
from travel.avia.flight_status_fetcher.sources.sever_aero.schedule import Schedule, Subject  # noqa
from travel.avia.flight_status_fetcher.sources.sever_aero.timetable import FlightRoute, Timetable

logger = logging.getLogger(__name__)


class SeverAero(ObjectWithStatistics):
    _flight_number_re = re.compile(r'.*?(\d+)\s*$')

    def __init__(self, avia_backend: AviaBackend, subjects_xml: str, timetable_xml: str):
        super().__init__()
        self._clear_statistics()
        self._avia_backend = avia_backend
        self._schedule = Schedule(xml_string=subjects_xml)
        self._timetable = Timetable(xml_string=timetable_xml)
        self.status_data_collector = StatusDataCollector()
        self.status_data_collector.partner = 'sever'
        self.status_data_collector.add_raw_data('\n===SUBJECT XML===\n\n\n')
        self.status_data_collector.add_raw_data(subjects_xml)
        self.status_data_collector.add_raw_data('\n===TIMETABLE XML===\n\n\n')
        self.status_data_collector.add_raw_data(timetable_xml)

    def get_iata_or_sirena_code_by_name_ru(self, name_ru) -> Optional[str]:
        """Найти код (IATA или Сирена) по названию филиала.

        В некоторых случаях название филиала - это название аэропорта,
        в некоторых - город, в котором аэропорт находится.se

        Логика поиска такова:
        1. Сначала ищем IATA код по названию аэропорта.
        2. Если не удалось, то ищем IATA код по названию города.
        3. Если не удалось, то ищем Сирена код по названию аэропорта.
        4. Если не удалось, то ищем Сирена код по названию города.

        :param name_ru: название филиала Аэропортов Севера
        :return: строка, если удалось найти код
        """
        name_ru = common.canonical_name_ru(name_ru)
        maybe_codes = [
            self._avia_backend.iata_by_aero_name.get(name_ru),
            self._avia_backend.iata_by_city_name.get(name_ru),
            self._avia_backend.sirena_by_aero_name.get(name_ru),
            self._avia_backend.sirena_by_city_name.get(name_ru),
        ]
        for maybe_code in maybe_codes:
            if maybe_code:  # нас устраивает только непустая строка, проверяем на None и на пустую строку
                return maybe_code
        logger.error('Cannot get IATA or Sirena code for subject name "%s"', name_ru)

    def get_iata_or_sirena_code_by_internal_code(self, code: str) -> Optional[str]:
        """Найти код (IATA или Сирена) по внутреннему идентификатору филиала.

        :param code: внутренний идентификатор филиала у Аэропортов Севера
        :return: IATA или Сирена код
        """
        subject = self._schedule.subjects_by_code.get(code)
        if subject is not None:
            return self.get_iata_or_sirena_code_by_name_ru(subject.name_ru)
        logger.error('Cannot find subject by code "%s"', code)

    @staticmethod
    def get_airline_code_by_route_name(route_name: str) -> Optional[str]:
        route_name = route_name.upper().strip()

        if route_name.startswith('ДРУ'):  # хак для Алросы
            return '6R'

        return route_name[:2]

    def get_airline_id_by_route_name(self, route_name: str) -> Optional[int]:
        airline_code = self.get_airline_code_by_route_name(route_name)
        if not airline_code:
            return None
        return self._avia_backend.airline_id_by_sirena.get(airline_code) or self._avia_backend.airline_id_by_iata.get(
            airline_code
        )

    def get_flight_number_by_route_name(self, route_name: str) -> Optional[str]:
        match = self._flight_number_re.search(route_name)
        if match:
            return match.group(1)
        else:
            logger.error('Cannot find out flight number from route name "%s".', route_name)

    @staticmethod
    def _check_substring_and_direction_consistency(direction: str, substr: str) -> bool:
        if direction == const.Direction.ARRIVAL.value:
            return substr in {'delay', 'landed', 'cancelled', 'on time'}

        if direction == const.Direction.DEPARTURE.value:
            return substr in {'delay', 'cancelled', 'on time', 'departed'}

        raise ValueError('Unknown direction {}'.format(direction))

    def _get_status_and_time_actual(
        self,
        route: FlightRoute,
        is_linked_route: bool = False,
    ) -> Tuple[Optional[str], Optional[dt.datetime]]:
        route_state_ru_lower = route.state_ru.lower()
        route_state_en_lower = route.state_en.lower()

        if 'совмещен с' in route_state_ru_lower:
            original_route = self._timetable.code_share_routes.get(route)
            if original_route is not None and not is_linked_route:
                return self._get_status_and_time_actual(original_route, True)
            else:
                logger.error('Cannot find original route for route %s', route)

        status_by_name_en_substring = {
            'delay': const.FlightStatus.STATUS_DELAY.value,
            'departed': const.FlightStatus.STATUS_DEPARTED.value,
            'landed': const.FlightStatus.STATUS_ARRIVED.value,
            'cancelled': const.FlightStatus.STATUS_CANCELLED.value,
            'on time': const.FlightStatus.STATUS_WAIT.value,
        }
        for substr, status in status_by_name_en_substring.items():
            if substr in route_state_en_lower:
                if self._check_substring_and_direction_consistency(route.direction, substr):
                    return status, route.time_actual
                else:
                    return const.FlightStatus.STATUS_UNKNOWN_VALUE.value, None

        return route.state_en if route.state_en else const.FlightStatus.STATUS_NO_DATA_VALUE.value, route.time_actual

    def route_to_status(self, route: FlightRoute, message_id, received_at) -> Optional[Status]:
        status, time_actual = self._get_status_and_time_actual(route)
        airport = self.get_iata_or_sirena_code_by_internal_code(route.subject_code)
        if not airport:
            logger.error(
                "Invalid route: internal airport code `%s` is not found: %s",
                route.subject_code,
                self._schedule.subjects_by_code.items(),
            )
            return None

        other_airport = self.get_iata_or_sirena_code_by_name_ru(route.name_ru)

        route_point_from = None
        route_point_to = None
        if route.direction == 'arrival':
            route_point_from = other_airport
            route_point_to = airport
        if route.direction == 'departure':
            route_point_from = airport
            route_point_to = other_airport

        return Status(
            message_id=message_id,
            received_at=received_at,
            airport=airport.upper(),
            airline_id=self.get_airline_id_by_route_name(route.route_name),
            airline_code=self.get_airline_code_by_route_name(route.route_name),
            flight_number=self.get_flight_number_by_route_name(route.route_name),
            flight_date=route.time_plan.date(),
            direction=route.direction,
            time_actual=time_actual,
            time_scheduled=route.time_plan,
            status=status,
            gate=None,
            terminal=None,
            check_in_desks=None,
            baggage_carousels=None,
            source=const.STATUS_SOURCE_AIRPORT,
            route_point_from=route_point_from,
            route_point_to=route_point_to,
        )

    def _get_statuses(self) -> Iterable[Status]:
        message_id = self.status_data_collector.message_id
        received_at = int(datetime.now().timestamp())
        self.status_data_collector.received_at = received_at
        for route in self._timetable.routes:
            status = self.route_to_status(route, message_id, received_at)
            if status and status.time_actual:
                self._last_run_statistics['success'] += 1
                yield status

    def collect_statuses(self) -> StatusDataPack:
        logger.info('Updating sever aero airport status')
        try:
            self.status_data_collector.statuses = list(self._get_statuses())
        except Exception as e:
            self.status_data_collector.error = 'Sever aero airport error: {}'.format(e)
            self._last_run_statistics['failure'] += 1
            logger.exception('Error working with statuses')
        finally:
            logger.info('Done updating sever aero airport status')
            return self.status_data_collector.status_data_pack()

    def _clear_statistics(self):
        self._last_run_statistics = {
            'success': 0,
            'failure': 0,
        }
