"""Парсер для файла со списком филиалов и расписанием маршрутов от Аэропортов Севера."""
import logging
from typing import NamedTuple, Iterator

from parsel import Selector

__all__ = ['Subject', 'Schedule']

logger = logging.getLogger(__name__)

# филиал Аэропортов Севера
Subject = NamedTuple(
    'Subject',
    [  # филиал - это один из аэропортов, находящихся под управлением Аэропортов Севера
        ('code', str),  # внутренний идентификатор филиала
        # Название филиала на русском. Для одиних филиалов это - название аэропорта, для других - название города.
        ('name_ru', str),
        # Название филиала на английском. Аналогично русскому названию.
        ('name_en', str),
    ],
)


class Schedule:
    def __init__(self, xml_string: str):
        self._root_selector = Selector(text=xml_string, type='xml')

        self._subjects_by_code = {}
        self._subjects_by_name_ru = {}

        for subject in self._parse_subjects():
            self._save_subject(subject)

    @property
    def subjects_by_code(self):
        return self._subjects_by_code

    @property
    def subjects_by_name_ru(self):
        return self._subjects_by_name_ru

    def _save_subject(self, subject: Subject):
        if subject.code in self._subjects_by_code:
            logger.error(
                'Duplicate subject code for subjects "%s" and "%s"',
                subject,
                self._subjects_by_code[subject.code],
            )
        self._subjects_by_code[subject.code] = subject
        if subject.name_ru in self._subjects_by_name_ru:
            logger.error(
                'Duplicate Russian name for subjects "%s" and "%s"',
                subject,
                self._subjects_by_name_ru[subject.name_ru],
            )
        self._subjects_by_name_ru[subject.name_ru] = subject

    def _parse_subjects(self) -> Iterator[Subject]:
        logger.debug('Parsing subjects...')
        for subject_sel in self._root_selector.xpath('/SUBJECT_LIST/SUBJECT'):
            code = subject_sel.xpath('CODE/text()').extract_first()
            name_ru = subject_sel.xpath('NAME[@LANG="RU"]/text()').extract_first()
            name_en = subject_sel.xpath('NAME[@LANG="EN"]/text()').extract_first()
            subject = Subject(code, name_ru, name_en)
            logger.debug(subject)
            yield subject
