"""Парсер для табло Аэропортов Севера."""
import datetime as dt
import logging
import re
from typing import NamedTuple, Iterator, Optional, List, Dict

from parsel import Selector

from travel.avia.flight_status_fetcher import const

logger = logging.getLogger(__name__)

FlightRoute = NamedTuple(
    'Route',
    [  # рейс на табло аэропорта
        ('subject_code', str),  # внутренний идентификатор филиала, к табло которого относится этот рейс
        # название маршрута в формате <Сирена код авиакомпании><номер маршрута> (слитно)
        ('route_name', str),  # например: ЯК418
        ('direction', str),  # направление: departure или arrival
        ('name_ru', str),  # аэропорт прибытия для вылетающих рейсов. Аэропорт отправления для прибывающих рейсов
        ('name_en', str),  # аналогично name_ru, но на английском
        ('state_ru', str),  # статус на русском
        ('state_en', str),  # статус на английском
        ('time_plan', Optional[dt.datetime]),  # запланированное время вылета/прибытия
        ('time_actual', Optional[dt.datetime]),  # реальное время прибытия
    ],
)


class Timetable:
    _find_time_re = re.compile(r'.*?(\d{1,2}):(\d{2})')
    _find_date_re = re.compile(r'.*?\((\d{2})\.(\d{2})\)')

    _find_route_name_re = re.compile(r'.*?([А-Яа-яЁё]+\d+)')

    @staticmethod
    def _fix_xml(xml_string: str):
        return xml_string.replace('<?xml version="1.0"?>', '<?xml version="1.0"?>\n<ROOT>') + '</ROOT>'

    def __init__(self, xml_string: str):
        self._root_selector = Selector(text=self._fix_xml(xml_string), type='xml')

        self._routes = []
        self._routes_by_name = {}
        self._code_share_routes = {}

        for route in self._parse_routes():
            self._save_route(route)

        self._link_routes()

    @property
    def routes(self) -> List[FlightRoute]:
        return self._routes

    @property
    def routes_by_name(self) -> Dict[str, FlightRoute]:
        return self._routes_by_name

    @property
    def code_share_routes(self) -> Dict[FlightRoute, FlightRoute]:
        """Словарь, в котором по дополнительному коду рейса можно найти основной."""
        return self._code_share_routes

    @classmethod
    def _datetime_from_date_time(cls, date: dt.date, time: dt.time) -> dt.datetime:
        naive_dt = dt.datetime(
            year=date.year,
            month=date.month,
            day=date.day,
            hour=time.hour,
            minute=time.minute,
        )
        return naive_dt

    def _parse_time_plan(self, time_plan_sel: Selector) -> Optional[dt.datetime]:
        date_string = time_plan_sel.xpath('@DATE').extract_first()
        time_string = time_plan_sel.xpath('@TIME').extract_first()
        error_message = 'Unable to parse datetime from %s. Date string: %s Time string: %s'
        try:
            date = dt.datetime.strptime(date_string, '%Y-%m-%d').date()
            time = dt.datetime.strptime(time_string, '%H:%M').time()
            return self._datetime_from_date_time(date, time)
        except ValueError:
            logging.exception(error_message, time_plan_sel, date_string, time_string)
        except TypeError:
            logging.exception(error_message, time_plan_sel, date_string, time_string)

    def _parse_time_actual(self, state_en: str, time_plan: Optional[dt.datetime]) -> Optional[dt.datetime]:
        if time_plan is None:
            return None

        date = time_plan.date()
        time = time_plan.time()

        match_date = self._find_date_re.match(state_en)
        match_time = self._find_time_re.match(state_en)

        if match_date:
            month = int(match_date.group(2))
            day = int(match_date.group(1))
            year = date.year if month >= date.month else date.year + 1
            date = dt.date(year=year, month=month, day=day)

        if match_time:
            hour = int(match_time.group(1))
            minute = int(match_time.group(2))
            time = dt.time(hour=hour, minute=minute)

        return self._datetime_from_date_time(date, time)

    def _parse_route_entity(self, route_entity_sel: Selector, subject_code: str, direction: str) -> FlightRoute:
        state_ru = route_entity_sel.xpath('STATE[@LANG="RU"]/text()').extract_first('')
        state_en = route_entity_sel.xpath('STATE[@LANG="EN"]/text()').extract_first('')

        time_plan = self._parse_time_plan(route_entity_sel.xpath('TIME_PLAN'))
        time_actual = self._parse_time_actual(state_en, time_plan)

        return FlightRoute(
            subject_code=subject_code,
            route_name=route_entity_sel.xpath('ROUTE_NAME/text()').extract_first(''),
            direction=direction,
            name_ru=route_entity_sel.xpath('NAME[@LANG="RU"]/text()').extract_first(''),
            name_en=route_entity_sel.xpath('NAME[@LANG="EN"]/text()').extract_first(''),
            state_ru=state_ru,
            state_en=state_en,
            time_plan=time_plan,
            time_actual=time_actual,
        )

    def _save_route(self, route: FlightRoute):
        logger.debug('Saving route %s', route.__repr__())
        if 'deleted' not in route.name_en.lower():
            self._routes.append(route)
            self._routes_by_name.setdefault(route.route_name, [])
            self._routes_by_name[route.route_name].append(route)

    def _parse_routes(self) -> Iterator[FlightRoute]:
        for arrival_list_sel in self._root_selector.xpath('//ARRIVAL_LIST'):
            subject_code = arrival_list_sel.xpath('@subject').extract_first()
            for route_entity_sel in arrival_list_sel.xpath('ROUTE_ENTITY'):
                yield self._parse_route_entity(route_entity_sel, subject_code, const.Direction.ARRIVAL.value)
        for departure_list_sel in self._root_selector.xpath('//DEPARTURE_LIST'):
            subject_code = departure_list_sel.xpath('@subject').extract_first()
            for route_entity_sel in departure_list_sel.xpath('ROUTE_ENTITY'):
                yield self._parse_route_entity(route_entity_sel, subject_code, const.Direction.DEPARTURE.value)

    def _link_routes(self):
        for route in self._routes:
            if 'совмещен с' in route.state_ru.lower():
                logger.debug('%s is a linked route', route.__repr__())
                match_name = self._find_route_name_re.match(route.state_ru)
                if match_name:
                    maybe_name = match_name.group(1)
                    logger.debug('Route with state_ru %s may be linked to %s', route.state_ru.lower(), maybe_name)
                    if maybe_name == route.route_name:
                        logger.debug('Route %s seems to be linked to itself', route)
                        continue
                    linked_routes = self._routes_by_name.get(maybe_name, [])
                    if len(linked_routes) > 1:
                        linked_routes = [
                            linked_route
                            for linked_route in linked_routes
                            if (
                                route.time_plan.date() == linked_route.time_plan.date()
                                and route.name_ru == linked_route.name_ru
                            )
                        ]
                    if len(linked_routes) > 0:
                        logger.debug('Linking route %s to %s', route, linked_routes[0])
                        self._code_share_routes[route] = linked_routes[0]
