import logging
from datetime import datetime
from json.decoder import JSONDecodeError
from typing import Dict, Optional

import requests.auth
from retrying import retry

from travel.avia.library.python.proxy_pool.proxy_pool import ProxyPool

from travel.avia.flight_status_fetcher.library.airport_importer import AirportImporter
from travel.avia.flight_status_fetcher.library.flight_number_parser import FlightNumber, FlightNumberParser
from travel.avia.flight_status_fetcher.library.raw_data import StatusDataCollector, StatusDataPack
from travel.avia.flight_status_fetcher.services.status import Status


def _retry_on_exception(exc):
    return isinstance(
        exc,
        (JSONDecodeError, requests.exceptions.ConnectionError, requests.exceptions.ChunkedEncodingError),
    )


class SVOFetcher:
    """Class for fetching raw flights from SVO"""

    DATETIME_FORMAT = '%Y-%m-%dT%H:%M:%SZ'
    API_BATCH_MAX_SIZE = 500

    CONNECT_TIMEOUT = 60
    READ_TIMEOUT = 60

    def __init__(self, api_url, api_oauth_token, logger, proxy_pool: Optional[ProxyPool] = None):
        self._api_url = api_url
        self._api_oauth_token = api_oauth_token
        self._logger = logger
        self._proxy_pool = proxy_pool

    @retry(
        retry_on_exception=_retry_on_exception,
        stop_max_attempt_number=3,
        wait_exponential_multiplier=100,
        wait_exponential_max=500,
    )
    def _get_batch_by_offset(self, from_date, to_date, offset):
        fetching_params = {
            'limit': self.API_BATCH_MAX_SIZE,
            'offset': offset,
            'from': from_date.strftime(self.DATETIME_FORMAT),
            'till': to_date.strftime(self.DATETIME_FORMAT),
        }
        try:
            proxies = None
            if self._proxy_pool:
                proxies = self._proxy_pool.get_proxy().get_requests_proxies()
            response = requests.get(
                self._api_url,
                params=fetching_params,
                headers=self._get_auth_header(),
                timeout=(self.CONNECT_TIMEOUT, self.READ_TIMEOUT),
                proxies=proxies,
            )
            return response.json()
        except JSONDecodeError:
            self._logger.exception(
                'Can\'t fetch flights batch: %s. Got content: %s', response.request.url, response.content
            )
            raise
        except Exception:
            self._logger.exception('Can\'t fetch flights batch with params %s', fetching_params)
            raise

    def _get_auth_header(self):
        return {'Authorization': 'Bearer {}'.format(self._api_oauth_token)}

    @staticmethod
    def _check_batch(batch):
        if not batch or 'error' in batch or not isinstance(batch, list):
            raise RuntimeError('Bad SVO API Response %s' % batch)

    def fetch(self, from_date, to_date, raw_data_collector: StatusDataCollector):
        current_offset = 0
        while True:
            self._logger.info('Current offset: %d', current_offset)
            batch_json = self._get_batch_by_offset(from_date, to_date, current_offset)
            raw_data_collector.add_raw_data('\n===JSON AT %d===\n\n\n' % current_offset)
            raw_data_collector.add_raw_data(batch_json)
            self._check_batch(batch_json)
            yield from batch_json

            current_offset += self.API_BATCH_MAX_SIZE
            if len(batch_json) < self.API_BATCH_MAX_SIZE:
                return


class SVOFlightHelper:
    AD = 'ad'
    BAGGAGE_BELT_ID = 'baggageBeltId'
    CHECKIN_ID = 'checkInId'
    CO = 'co'
    FLIGHT = 'flight'
    GATE_ID = 'gateId'
    STATUS_CODE = 'statusCode'
    STATUS_DETAILED_ENG = 'statusDetailedEng'
    T_AT = 'tAt'
    T_ET = 'tEt'
    T_ST = 'tSt'
    TERMINAL = 'terminal'

    def __init__(self, raw_flight: dict):
        self._raw_flight: dict = raw_flight

    @classmethod
    def get_mar(cls, flight, i: int):
        return flight.get('mar{}'.format(i)) or flight.get('mar{}_id'.format(i))

    @classmethod
    def has_mar(cls, flight, i: int):
        return 'mar{}'.format(i) in flight


class SVOImporter(AirportImporter):
    AIRPORT = 'SVO'
    DATETIME_FORMAT = '%Y-%m-%dT%H:%M:%SZ'
    UNKNOWN_ROUTE_POINT_CODES = {'XXX'}

    def __init__(
        self,
        from_date,
        to_date,
        fetcher,
        logger: logging.Logger,
        flight_number_parser: FlightNumberParser,
    ):
        super().__init__(logger, flight_number_parser)
        self.from_date = from_date
        self.to_date = to_date
        self._fetcher = fetcher
        self.status_data_collector.partner = self.AIRPORT

    @staticmethod
    def _get_status(flight):
        status_code = flight.get(SVOFlightHelper.STATUS_CODE)
        status_detailed_eng = flight.get(SVOFlightHelper.STATUS_DETAILED_ENG)
        if status_code in ['16', '31', '51', '6', '8']:
            return 'departed'
        if status_code == '1' or (status_detailed_eng and 'cancelled' in status_detailed_eng.lower()):
            return 'cancelled'

        return {
            '2': 'delay',
            '20': 'boarding',
            '17': 'registration',
            '12': 'arrived',
        }.get(status_code, status_detailed_eng)

    @staticmethod
    def _parse_direction(direction):
        return {'D': 'departure', 'A': 'arrival'}[direction]

    def _get_points(self, flight, direction: str) -> (Optional[str], Optional[str]):
        i = 1
        while SVOFlightHelper.has_mar(flight, i + 1):
            point_from = SVOFlightHelper.get_mar(flight, i)
            point_to = SVOFlightHelper.get_mar(flight, i + 1)
            if not point_to:
                break
            if direction == 'departure' and point_from == self.AIRPORT:
                return point_from, point_to
            elif direction == 'arrival' and point_to == self.AIRPORT:
                return point_from, point_to
            i += 1

        return None, None

    def _make_statuses(self, flight: dict, flight_numbers: Dict[str, FlightNumber], message_id, received_at):
        direction = self._parse_direction(flight[SVOFlightHelper.AD])
        terminal = flight[SVOFlightHelper.TERMINAL]
        scheduled_time = self._utc_to_msk(self._parse_datetime(flight[SVOFlightHelper.T_ST]))
        actual_time = self._get_actual_time(
            self._utc_to_msk(self._parse_datetime(flight[SVOFlightHelper.T_ET], allow_none=True)),
            self._utc_to_msk(self._parse_datetime(flight[SVOFlightHelper.T_AT], allow_none=True)),
        )
        gate = None if direction == 'arrival' else flight[SVOFlightHelper.GATE_ID]
        baggage_carousels = flight.get(SVOFlightHelper.BAGGAGE_BELT_ID) if direction == 'arrival' else None
        raw_status = flight.get(SVOFlightHelper.STATUS_DETAILED_ENG)
        diverted = self._is_diverted(raw_status)
        route_point_from, route_point_to = self._get_points(flight, direction)

        if route_point_from in self.UNKNOWN_ROUTE_POINT_CODES or route_point_to in self.UNKNOWN_ROUTE_POINT_CODES:
            return

        for flight_number in self._extract_flight_numbers(flight):
            if not flight_numbers[flight_number].company_code:
                continue

            yield Status(
                message_id=message_id,
                received_at=received_at,
                direction=direction,
                airport=self.AIRPORT,
                airline_id=flight_numbers[flight_number].company_id,
                airline_code=flight_numbers[flight_number].company_code,
                flight_number=flight_numbers[flight_number].number,
                flight_date=scheduled_time.date(),
                status=self._get_status(flight) or 'no-data',
                terminal=terminal or None,
                gate=gate,
                time_actual=actual_time,
                time_scheduled=scheduled_time,
                check_in_desks=flight.get(SVOFlightHelper.CHECKIN_ID) or None,
                baggage_carousels=baggage_carousels or None,
                source=self.SOURCE,
                diverted=diverted,
                diverted_airport_iata=self._diverted_airport(diverted, raw_status),
                route_point_from=route_point_from,
                route_point_to=route_point_to,
            )

    @staticmethod
    def _is_diverted(status_str):
        return isinstance(status_str, str) and 'diverted' in status_str.lower()

    @staticmethod
    def _diverted_airport(diverted, status_str):
        if not diverted:
            return None
        return status_str.replace('Diverted to', '').strip()

    def _get_api_flights(self, from_date, to_date):
        yield from self._fetcher.fetch(from_date, to_date, self.status_data_collector)

    @staticmethod
    def _extract_flight_numbers(flight: dict, **kwargs):
        airline_code = flight[SVOFlightHelper.CO]
        flight_number = flight[SVOFlightHelper.FLIGHT]

        # Если Шереметьево не прислал код авиакомпании, отдаем право определения в shared-flights
        if airline_code == 'XX':
            airline_code = ''

        return ['{} {}'.format(airline_code, flight_number).lstrip()]

    def _build_statuses(self, from_date, to_date):
        flights = list(self._get_api_flights(from_date, to_date))
        self._last_run_statistics['datasize'] = len(flights)

        flight_numbers = self._parse_flight_numbers(flights)

        message_id = self.status_data_collector.message_id
        received_at = int(datetime.now().timestamp())
        self.status_data_collector.received_at = received_at

        for f in flights:
            try:
                for status in self._make_statuses(f, flight_numbers, message_id, received_at):
                    yield status
                    self._last_run_statistics['success'] += 1
            except Exception:
                self.logger.exception(u'Could not transform flight %s into Status', f)
                self._last_run_statistics['failure'] += 1

    def collect_statuses(self) -> StatusDataPack:
        self.logger.info(
            'SVO update started from %s to %s',
            self.from_date.strftime('%Y-%m-%d'),
            self.to_date.strftime('%Y-%m-%d'),
        )
        self._clear_statistics()
        try:
            self.status_data_collector.statuses = list(self._build_statuses(self.from_date, self.to_date))
        except Exception as e:
            self.status_data_collector.error = 'SVO update error: {}'.format(e)
            self.logger.exception('SVO update error')
        finally:
            self.logger.info('SVO update finished')
            return self.status_data_collector.status_data_pack()
