""" Flight status importer from VKO airport """
import logging
import re
from datetime import datetime
from urllib.parse import urljoin

from lxml import etree
from typing import Dict, Optional

import requests.auth

from travel.avia.library.python.proxy_pool.proxy_pool import ProxyPool

from travel.avia.flight_status_fetcher.library.airport_importer import AirportHTTPImporter
from travel.avia.flight_status_fetcher.library.flight_number_parser import FlightNumber, FlightNumberParser
from travel.avia.flight_status_fetcher.library.raw_data import StatusDataPack
from travel.avia.flight_status_fetcher.services.status import Status
from travel.avia.flight_status_fetcher.utils.safe_xml_parser import safe_xml_fromstring
from travel.avia.flight_status_fetcher.utils.xml import xml_to_dict


class VKOImporter(AirportHTTPImporter):
    AIRPORT = 'VKO'
    DATETIME_FORMAT = '%d.%m.%Y %H:%M:%S'
    RE_LAST_URL = re.compile(r'<a href="([\d_\-]+/)">.*\n</pre>')

    def __init__(
        self,
        data_url,
        data_filename,
        data_user,
        data_password,
        retry_settings,
        logger: logging.Logger,
        flight_number_parser: FlightNumberParser,
        request_timeout,
        proxy_pool: Optional[ProxyPool] = None,
    ):
        super().__init__(retry_settings, logger, flight_number_parser, proxy_pool)
        self._data_url = data_url
        self._data_filename = data_filename
        self._data_user = data_user
        self._data_password = data_password
        self._request_timeout = request_timeout
        self.status_data_collector.partner = self.AIRPORT

    @staticmethod
    def _parse_direction(direction):
        return {'0': 'departure', '1': 'arrival'}[direction]

    @staticmethod
    def _get_status_from_remarks(remarks):
        if 'cancel' in remarks.lower():
            return 'cancelled'
        if 'landed' in remarks.lower():
            return 'arrived'
        if 'departed' in remarks.lower():
            return 'departed'
        if 'delayed' in remarks.lower():
            return 'delay'

        return remarks

    @staticmethod
    def _extract_flight_numbers(flight, **kwargs):
        company_code = flight.get('AIRLINE_IATA', flight.get('AIRLINE_VNT'))
        if company_code:
            return ['{} {}'.format(company_code, flight['FLIGHT_NUMBER_FOR_PASS'].split()[-1])]

        return [flight['FLIGHT_NUMBER_FOR_PASS']]

    def _prepare_check_in_desks(self, flight) -> Optional[str]:
        raw_check_in_desks = flight.get('CHECKIN_LIST_ACTUAL_FOR_PASS')
        if not raw_check_in_desks:
            return None
        try:
            check_in_desks = (d for d in raw_check_in_desks.replace(')', ',').replace('(', ',').split(',') if d)
            check_in_desks = (self._rstrip_alphabetic_characters(d) for d in check_in_desks)
            check_in_desks = sorted(check_in_desks, key=lambda x: int(x))
            return self._join_check_in_desks(check_in_desks)
        except Exception:
            self.logger.exception('Can\'t parse check-in desks of flight: %s', flight)

    def _make_statuses(self, flight, flight_numbers: Dict[str, FlightNumber], message_id, received_at):
        scheduled_time = self._parse_datetime(flight['ST'])
        expected_time = self._parse_datetime(flight.get('ET_FOR_PASS'), allow_none=True)
        actual_time = self._parse_datetime(flight.get('AT_FOR_PASS'), allow_none=True)
        direction = self._parse_direction(flight['BOUND']) or None
        route_point = flight.get('FLS_IATA') or flight.get('FLS_FULL_NAME_RUS')
        route_point_from = route_point if direction == 'arrival' else self.AIRPORT
        route_point_to = self.AIRPORT if direction == 'arrival' else route_point

        for flight_number in self._extract_flight_numbers(flight):
            if not flight_numbers[flight_number].company_code:
                continue

            yield Status(
                message_id=message_id,
                received_at=received_at,
                airport=self.AIRPORT,
                airline_id=flight_numbers[flight_number].company_id,
                airline_code=flight_numbers[flight_number].company_code,
                flight_number=flight_numbers[flight_number].number,
                flight_date=scheduled_time.date() or None,
                direction=direction,
                status=self._get_status_from_remarks(flight.get('VNUKOVO_EN_STATUS', '')) or 'no-data',
                gate=flight.get('GATES_LIST_FOR_PASS') or None,
                terminal=flight['TERM'] or None,
                time_actual=self._get_actual_time(expected_time, actual_time) or None,
                time_scheduled=scheduled_time or None,
                check_in_desks=self._prepare_check_in_desks(flight) or None,
                baggage_carousels=flight.get('BELT') or None,
                source=self.SOURCE,
                route_point_from=route_point_from,
                route_point_to=route_point_to,
            )

    def _download_xml(self):
        return self._curl(
            urljoin(self._data_url, self._data_filename),
            auth=requests.auth.HTTPBasicAuth(self._data_user, self._data_password),
            timeout=self._request_timeout,
        )

    def _build_statuses(self, xml=None):
        if not xml:
            xml = self._download_xml()
        self.status_data_collector.add_raw_data(xml, 'windows-1251')
        try:
            tree = safe_xml_fromstring(xml)
        except etree.XMLSyntaxError as e:
            self.logger.error('Bad XML from vko: %s', e)
            raise

        flights = list(map(xml_to_dict, tree.xpath('//ROWSET/ROW')))
        flight_numbers = self._parse_flight_numbers(flights)

        message_id = self.status_data_collector.message_id
        received_at = int(datetime.now().timestamp())
        self.status_data_collector.received_at = received_at

        for flight in flights:
            try:
                for status in self._make_statuses(flight, flight_numbers, message_id, received_at):
                    yield status
                    self._last_run_statistics['success'] += 1
            except Exception:
                self.logger.exception('Could not transform flight %s into status', flight)
                self._last_run_statistics['failure'] += 1

    def collect_statuses(self, xml=None) -> StatusDataPack:
        self.logger.info('VKO update started')
        self._clear_statistics()
        try:
            self.status_data_collector.statuses = list(self._build_statuses(xml))
        except Exception as e:
            self.status_data_collector.error = 'VKO update error: {}'.format(e)
            self.logger.exception('VKO update error')
        finally:
            self.logger.info('VKO update finished')
            return self.status_data_collector.status_data_pack()
