import logging
from concurrent.futures import ThreadPoolExecutor, TimeoutError, as_completed
from typing import List, Iterable

from travel.library.python.logbroker.writer import get_deploy_source_id
from travel.avia.flight_status_fetcher.settings.services import logbroker
from travel.avia.flight_status_fetcher.library.raw_data import StatusDataPack
from travel.avia.flight_status_fetcher.services.logbroker import LogbokerStatusProducer, RawDataLogbrokerProducer
from travel.avia.flight_status_fetcher.services.status import IStatusUpdater, IStatus
from travel.avia.flight_status_fetcher.services.status import UpdateStatusError

UPDATE_STATUS_TIMEOUT = 60

logger = logging.getLogger(__name__)


def get_status_updaters(airport: str) -> List[IStatusUpdater]:
    status_updaters = []

    if logbroker.LOGBROKER_TOKEN and logbroker.LOGBROKER_STATUS_TOPICS:
        for topic in logbroker.LOGBROKER_STATUS_TOPICS:
            if not topic:
                continue
            try:
                status_updaters.append(
                    LogbokerStatusProducer(
                        token=logbroker.LOGBROKER_TOKEN,
                        topic=topic,
                        producer_id=airport,
                        source_id_generator=get_deploy_source_id,
                    ),
                )
            except ValueError:
                logger.exception('Cannot create logbroker producer')
    return status_updaters


def update_statuses(statuses: Iterable[IStatus], airport: str):
    statuses = list(statuses)
    status_updaters = get_status_updaters(airport)
    if len(status_updaters) == 0:
        logger.warning('Consumer list is empty. %d statuses won\'t be sent', len(statuses))
        return
    futures = []
    e = ThreadPoolExecutor(max_workers=len(status_updaters) * 4)
    try:
        for consumer in status_updaters:
            futures.append(e.submit(consumer.update_status, statuses))

        for future in as_completed(futures, timeout=UPDATE_STATUS_TIMEOUT):
            try:
                status, req_id = future.result()
                logger.info('Request %r completed with status %r', req_id, status)
            except UpdateStatusError as exc:
                logger.exception('Request %r completed with exception', exc.req_id)
            except Exception:
                logger.exception('Request completed with unknown exception')
    except TimeoutError:
        logger.exception('Some operations timed out')
    except Exception:
        logger.exception('Unknown exception during request')
    finally:
        e.shutdown(wait=False)


def send_raw_data(raw_data: StatusDataPack, airport: str):
    if logbroker.LOGBROKER_TOKEN and logbroker.LOGBROKER_RAW_DATA_TOPICS:
        for topic in logbroker.LOGBROKER_RAW_DATA_TOPICS:
            try:
                raw_data_logbroker_sender = RawDataLogbrokerProducer(
                    topic=topic,
                    token=logbroker.LOGBROKER_TOKEN,
                    producer_id=airport,
                    source_id_generator=get_deploy_source_id,
                )
                raw_data_logbroker_sender.send_raw_data(raw_data)
            except Exception:
                logger.exception('Cannot send status with raw data')
    else:
        logger.warning(
            'Raw data is not sent, because logbroker is not configured. %d. %s',
            len(raw_data.data),
            raw_data.data[:100],
        )
