import re
import datetime
import logging
import itertools
from abc import ABC, abstractmethod
from typing import Iterable, Optional

import pytz

from travel.avia.flight_status_fetcher.library.flight_number_parser import FlightNumberParser

TRAILING_ALPHABETIC_REGEXP = re.compile(r'(.+?)[A-z]+$')


class ImportHelper(ABC):
    def __init__(self, logger: logging.Logger, flight_number_parser: FlightNumberParser, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.logger = logger
        self._clear_statistics()
        self.flight_number_parser = flight_number_parser

    @property
    @abstractmethod
    def DATETIME_FORMAT(self) -> str:
        pass

    @staticmethod
    @abstractmethod
    def _extract_flight_numbers(flight, **kwargs) -> Iterable[str]:
        pass

    def _parse_flight_numbers(self, flights, **kwargs):
        flight_numbers = set(itertools.chain(*[self._extract_flight_numbers(flight, **kwargs) for flight in flights]))
        parsed_flight_numbers = self.flight_number_parser.parse_list(flight_numbers)
        for raw_flight_number, flight_number in parsed_flight_numbers.items():
            if not flight_number.company_id:
                self._last_run_statistics['company_not_found'] = (
                    self._last_run_statistics.get(
                        'company_not_found',
                        0,
                    )
                    + 1
                )
                self.logger.warning(
                    'Company "{}" not found for "%s %s", raw "%s"'.format(flight_number.company_code),
                    flight_number.company_code,
                    flight_number.number,
                    raw_flight_number,
                )
        return parsed_flight_numbers

    @staticmethod
    def _get_actual_time(time_expected, time_actual):
        if time_actual:
            return time_actual
        if time_expected:
            return time_expected
        return None

    def _parse_datetime(self, dt_string, allow_none=False):
        if not dt_string:
            if allow_none:
                return None
            raise Exception("Datetime expected, got '{}'".format(dt_string))
        try:
            return datetime.datetime.strptime(dt_string, self.DATETIME_FORMAT)
        except Exception:
            self.logger.exception(
                'Could not parse datetime from %s. Expected format %s.',
                dt_string,
                self.DATETIME_FORMAT,
            )
            raise

    @staticmethod
    def _format_date(dt: datetime.datetime) -> str:
        return dt.strftime('%Y-%m-%d')

    @staticmethod
    def _format_datetime(dt: datetime.datetime) -> str:
        return dt.strftime('%Y-%m-%dT%H:%M:%S')

    @staticmethod
    def _utc_to_msk(utc_dt: Optional[datetime.datetime]) -> Optional[datetime.datetime]:
        if not utc_dt:
            return

        moscow_timezone = pytz.timezone('Europe/Moscow')
        local_dt = utc_dt.replace(tzinfo=pytz.utc).astimezone(moscow_timezone)
        return moscow_timezone.normalize(local_dt)

    def _clear_statistics(self):
        self._last_run_statistics = {
            'success': 0,
            'failure': 0,
            'company_not_found': 0,
            'datasize': 0,
        }

    def _join_check_in_desks(self, check_in_desks) -> Optional[str]:
        if not check_in_desks:
            return None
        try:
            check_in_desks = sorted(check_in_desks)
            interval_start = current = check_in_desks[0]
            result = []
            for d in check_in_desks:
                if int(current) + 1 >= int(d):
                    current = d
                    continue
                result.append('{}-{}'.format(interval_start, current) if interval_start != current else current)
                interval_start = current = d
            result.append('{}-{}'.format(interval_start, current) if interval_start != current else current)
            return ','.join(result)
        except Exception:
            self.logger.exception('Can\'t join check in desks: %s', check_in_desks)

    @staticmethod
    def _rstrip_alphabetic_characters(check_in_desk) -> str:
        return re.sub(TRAILING_ALPHABETIC_REGEXP, r'\1', check_in_desk)
