from typing import Dict, Any, Optional

from requests import Session
from requests.adapters import HTTPAdapter
from requests.auth import HTTPBasicAuth
from urllib3 import Retry
from zeep import Transport, Client

from travel.avia.library.python.proxy_pool.proxy_pool import ProxyPool


class ProxyTransport(Transport):
    def __init__(
        self,
        cache=None,
        timeout=300.0,
        operation_timeout=None,
        session=None,
        proxy_pool: Optional[ProxyPool] = None,
    ):
        super().__init__(cache, timeout, operation_timeout, session)
        self._proxy_pool = proxy_pool

    def _update_proxies(self):
        if self._proxy_pool:
            proxy = self._proxy_pool.get_proxy()
            self.session.proxies = proxy.get_requests_proxies()

    def get(self, address, params, headers):
        self._update_proxies()
        return super().get(address, params, headers)

    def post(self, address, message, headers):
        self._update_proxies()
        return super().post(address, message, headers)

    def _load_remote_data(self, url):
        self._update_proxies()
        return super()._load_remote_data(url)


def build_client(
    url: str,
    login: str,
    password: str,
    retry_settings: Dict[Any, Any],
    request_timeout: float,
    operation_timeout: float,
    proxy_pool: Optional[ProxyPool] = None,
):
    session = Session()
    retry = Retry(**retry_settings)
    session.mount(url, HTTPAdapter(max_retries=retry))
    if login and password:
        session.auth = HTTPBasicAuth(login, password)

    return Client(
        url,
        transport=ProxyTransport(
            timeout=request_timeout,
            operation_timeout=operation_timeout,
            session=session,
            proxy_pool=proxy_pool,
        ),
    )
