import logging
import time

import celery.signals

from travel.avia.flight_status_fetcher.settings.services import avia_backend as backend_settings, sqs, proxy_pool
from travel.avia.flight_status_fetcher.settings.sources import sever_aero as sever_aero_settings
from travel.avia.flight_status_fetcher.ftp_wrapper import FTPWrapper
from travel.avia.flight_status_fetcher.library.solomon import transform_dict_to_sensors, upload_to_solomon
from travel.avia.flight_status_fetcher.services.avia_backend import AviaBackend
from travel.avia.flight_status_fetcher.sources.sever_aero import SeverAero
from travel.avia.flight_status_fetcher.updaters import update_statuses, send_raw_data

logger = logging.getLogger(__name__)

broker = 'sqs://%s:%s@%s' % (sqs.SQS_ACCESS_KEY, sqs.SQS_SECRET_KEY, sqs.SQS_ENDPOINT)
app = celery.Celery('worker', broker=broker)
app.conf.broker_transport_options = {
    'is_secure': False,
    'region': sqs.SQS_REGION,
    'queue_name_prefix': sqs.SQS_QUEUE_NAME_PREFIX,
}
app.conf.worker_hijack_root_logger = False


@celery.signals.setup_logging.connect
def setup_logging(*args, **kwargs):
    """Не даём селери инициализировать своё логгирование, чтобы работали наши хэндлеры."""


@app.task
def update_sever_aero():
    start = time.time()
    logger.info('Sever Aero flights update started.')
    avia_backend = AviaBackend.get_instance(
        backend_settings.AVIA_BACKEND_BASE_URL,
        backend_settings.AVIA_BACKEND_RETRY,
        backend_settings.AVIA_BACKEND_TIMEOUT,
        backend_settings.AVIA_BACKEND_REFRESH_REFERENCE_TIMEOUT,
    )
    ftp = FTPWrapper(
        host=sever_aero_settings.SEVER_AERO_FTP_HOST,
        user=sever_aero_settings.SEVER_AERO_FTP_LOGIN,
        passwd=sever_aero_settings.SEVER_AERO_FTP_PASSWORD,
        timeout=sever_aero_settings.SEVER_AERO_FTP_TIMEOUT,
        retry_kwargs=sever_aero_settings.SEVER_AERO_FTP_RETRY,
        proxy_pool=proxy_pool.PARTNERS_PROXY_POOL,
    )

    try:
        modification_time = ftp.get_modification_time(sever_aero_settings.SEVER_AERO_FTP_TIMETABLE_PATH)
    except Exception:
        logger.exception('During getting modificaion time')
        modification_time = None

    sever_aero = SeverAero(
        avia_backend,
        subjects_xml=ftp.retrieve_string(
            sever_aero_settings.SEVER_AERO_FTP_SUBJECTS_PATH, sever_aero_settings.SEVER_AERO_XML_ENCODING
        ),
        timetable_xml=ftp.retrieve_string(
            sever_aero_settings.SEVER_AERO_FTP_TIMETABLE_PATH, sever_aero_settings.SEVER_AERO_XML_ENCODING
        ),
    )

    try:
        status_data_pack = sever_aero.collect_statuses()
        if status_data_pack is None:
            logger.error('Sever aero collector returned `None` status pack')
    except Exception:
        logger.exception('Sever aero collector failed')
        status_data_pack = None

    if status_data_pack is not None:
        update_statuses(status_data_pack.statuses, 'sever')
        send_raw_data(status_data_pack, 'sever')

    logger.info('Sever Aero flights update finished.')

    end = time.time()

    data = dict(sever_aero.statistics, elapsed=end - start)
    if modification_time:
        data['timetable_age'] = end - modification_time.timestamp()
    upload_to_solomon(
        {
            'source': 'sever_aero',
        },
        transform_dict_to_sensors(end, data),
    )
