package app

import (
	"net/http"

	"github.com/go-chi/chi/v5"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/flight_status_receiver/internal/collectors"
	"a.yandex-team.ru/travel/avia/flight_status_receiver/internal/config"
	"a.yandex-team.ru/travel/avia/flight_status_receiver/internal/delivery"
	"a.yandex-team.ru/travel/avia/flight_status_receiver/internal/handlers"
	"a.yandex-team.ru/travel/avia/flight_status_receiver/internal/pipeline"
	"a.yandex-team.ru/travel/avia/flight_status_receiver/pkg"
)

type App struct {
	PipelineManager pkg.PipelineManager
	Handlers        http.Handler
}

func NewAppP(conf config.Config) (App, error) {
	var nilApp App
	pm := configurePipelinesP(conf)
	h, err := appHandlers(pm)
	if err != nil {
		return nilApp, xerrors.Errorf("NewAppP: %w", err)
	}
	return App{
		pm, h,
	}, nil
}

func configurePipelinesP(conf config.Config) pkg.PipelineManager {
	cm := collectors.NewCollectorsManagerP()
	rawDataDelivery := delivery.NewLogbrokerDeliveryP(delivery.WriteTopicWithCreds{
		Endpoint: conf.Logbroker.Endpoint,
		Token:    conf.Logbroker.Token,
		Topic:    conf.Logbroker.RawDataTopic,
		SourceID: conf.Logbroker.SourceID,
	})
	deliverySlice := []pkg.Delivery{delivery.StdoutDelivery{}}
	deliverySlice = append(deliverySlice, statusTopicDeliveryForConfig(conf)...)
	statusDelivery := delivery.NewDuplicatorP(deliverySlice...)

	pm := pipeline.NewPipelineManagerP(cm, statusDelivery, rawDataDelivery)
	return pm
}

func statusTopicDeliveryForConfig(conf config.Config) []pkg.Delivery {
	var deliverySlice []pkg.Delivery
	for _, topic := range conf.Logbroker.StatusTopics {
		if len(topic) == 0 {
			continue
		}
		deliverySlice = append(
			deliverySlice,
			delivery.NewLogbrokerDeliveryP(delivery.WriteTopicWithCreds{
				Endpoint: conf.Logbroker.Endpoint,
				Token:    conf.Logbroker.Token,
				Topic:    topic,
				SourceID: conf.Logbroker.SourceID,
			}),
		)
	}
	return deliverySlice
}

func appHandlers(pipelineManager pkg.PipelineManager) (http.Handler, error) {
	mux := chi.NewMux()

	for _, handler := range handlers.Handlers {
		h, err := handler(pipelineManager)
		if err != nil {
			return nil, xerrors.Errorf("appHandlers: %w", err)
		}

		route := h.GetRoute()
		if route == "" || route[len(route)-1] != '/' {
			mux.Handle(route, h)
			mux.Handle(route+"/", h)
			route += "/"
		}
		mux.Handle(route+"*", h)
	}

	return mux, nil
}
