package collectors

import (
	"encoding/json"
	"regexp"
	"sort"
	"strconv"
	"strings"

	"github.com/gofrs/uuid"
	"golang.org/x/xerrors"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/flight_status_receiver/pkg"
	dir "a.yandex-team.ru/travel/avia/shared_flights/lib/go/direction"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
	"a.yandex-team.ru/travel/proto/avia/flight_status"
)

func AuroraCollectorP() pkg.Collector {
	if c, err := DefaultCollector(
		Aurora,
		pkg.DecoderFunc(AuroraDecoder),
	); err != nil {
		panic(err)
	} else {
		return c
	}

}

type AuroraMessage struct {
	pkg.AuroraMessage
}

func (m AuroraMessage) Validate() error {
	return nil
}

var statusConvertMap = map[string]string{
	"ожидается":               "wait",
	"расчетное":               "wait",
	"прибытие по расписанию":  "wait",
	"по расписанию":           "wait",
	"вылет по расписанию":     "wait",
	"ожидается прибытие":      "wait",
	"в пути":                  "wait",
	"вовремя":                 "wait",
	"ожидается по расписанию": "wait",
	"в полете":                "wait",
	"раннее прибытие":         "wait",

	"задерживается":             "delay",
	"задержка":                  "delay",
	"задержан":                  "delay",
	"задержка решение АК":       "delay",
	"позднее прибытие":          "delay",
	"задержка позднее прибытие": "delay",
	"задержка подготовка рейса": "delay",
	"задержка метеоусловия":     "delay",

	"вылетел":      "departed",
	"отправлен":    "departed",
	"рейс вылетел": "departed",

	"прибыл":                    "arrived",
	"приземлился":               "arrived",
	"рейс прибыл":               "arrived",
	"совершил посадку":          "arrived",
	"прибыл/выдача багажа":      "arrived",
	"произвел посадку":          "arrived",
	"выдача багажа":             "arrived",
	"прилетел":                  "arrived",
	"прилетел с задержкой":      "arrived",
	"прибыл в пункт назначения": "arrived",

	"отменен":      "cancelled",
	"отмена":       "cancelled",
	"рейс отменен": "cancelled",
	"отменён":      "cancelled",

	"идет регистрация":    "registration",
	"регистрация открыта": "registration",
	"регистрация":         "registration",

	"регистрация закрыта":   "registration-finished",
	"регистрация закончена": "registration-finished",
	"регистрация окончена":  "registration-finished",
	"регистр-ия оконч":      "registration-finished",
	"регистр. завершена":    "registration-finished",

	"посадка":           "boarding",
	"идет посадка":      "boarding",
	"посадка в самолет": "boarding",

	"посадка закончена": "boarding-finished",
	"посадка окончена":  "boarding-finished",
	"посадка завершена": "boarding-finished",

	"неизвестный или пустой статус": "unknown",
	"неизвестный статус":            "unknown",
	"статус неизвестен":             "unknown",

	"информация отсутствует": "no-data",
}
var statusConvertMapByDirection = map[dir.Direction]map[string]string{
	dir.DEPARTURE: {
		"руление": "boarding-finished",
	},
	dir.ARRIVAL: {},
}

var statusConvertMapRegexp = map[dir.Direction]map[string]*regexp.Regexp{
	dir.DEPARTURE: {
		"wait":                  regexp.MustCompile(`регистрация с .+, посадка с .+`),
		"registration":          regexp.MustCompile(`регистрация [\d\-]+`),
		"registration-finished": regexp.MustCompile(`регистрация закончена..+`),
	},
	dir.ARRIVAL: {},
}

var statusIgnoreRegexp = regexp.MustCompile(`Выход изменён на \d+`)

var flightNumberRegexp = regexp.MustCompile(`^(0*\d{1,4}).*$`)

var actualDatetimeRegexp = regexp.MustCompile(`(\d{4}-\d{2}-\d{2}T\d{2}:\d{2}(:\d{2})?)$`)
var actualTimeRegexp = regexp.MustCompile(`(\d{2}:\d{2}(:\d{2})?)$`)

var auroraCodeshareSplitRegexp = regexp.MustCompile(`[ \-]+`)

type flightNumber struct {
	airline   string
	number    string
	codeshare bool
}

func (m AuroraMessage) Normalize() ([]*flight_status.FlightStatus, error) {
	var statuses []*flight_status.FlightStatus
	for _, auroraAirport := range m.AuroraMessage {
		direction, err := dir.FromString(auroraAirport.Direction)
		if err != nil {
			return nil, xerrors.Errorf("<AuroraMessage>.Normalize: %w", err)
		}
		for _, auroraFlight := range auroraAirport.Flights {
			for _, flight := range flightNumbers(auroraFlight) {
				status, err := createStatus(
					flight,
					direction,
					auroraAirport,
					auroraFlight,
				)
				if err != nil {
					return nil, xerrors.Errorf("<AuroraMessage>.Normalize: %w", err)
				}
				statuses = append(statuses, status)
			}

		}
	}

	for i := range statuses {
		statusID, err := uuid.NewV4()
		if err != nil {
			return nil, xerrors.Errorf("<AuroraMessage>.Normalize: %w", err)
		}
		statuses[i].StatusId = statusID.String()
	}
	return statuses, nil
}

func flightNumbers(flight pkg.AuroraFlight) []flightNumber {
	var flights []flightNumber
	flights = append(flights, flightNumber{
		airline:   flight.Airline,
		number:    flight.FlightNumber,
		codeshare: false,
	})
	for _, codeshare := range flight.Codeshares {
		airline, number, err := parseFlightNumber(codeshare)
		if err != nil {
			logger.Logger().Error(
				"Cannot parse aurora codeshare",
				log.Error(err),
				log.Reflect("flight", flight),
			)
			continue
		}
		flights = append(flights, flightNumber{
			airline:   airline,
			number:    number,
			codeshare: true,
		})
	}
	return flights
}

func cleanupCarrierPrefixFromFlightNumber(carrier string, number string) (string, string) {
	if len(carrier) > 0 && len(number) > len(carrier) && strings.HasPrefix(number, carrier) {
		return carrier, number[len(carrier):]
	}
	return carrier, number
}

func parseFlightNumber(fn string) (string, string, error) {
	cc := auroraCodeshareSplitRegexp.Split(strings.Trim(fn, " "), -1)
	if len(cc) != 2 {
		return "", "", xerrors.Errorf("incorrect flight number format: %v", fn)
	}
	cc[0], cc[1] = cleanupCarrierPrefixFromFlightNumber(cc[0], cc[1])
	matches := flightNumberRegexp.FindStringSubmatch(cc[1])
	if len(matches) != 2 {
		return "", "", xerrors.Errorf(
			"can not match flight_number %s and airline_code %s from aurora",
			cc[1], cc[0],
		)
	}
	return cc[0], strings.TrimLeft(cc[1], "0"), nil
}

func createStatus(
	flightNumber flightNumber,
	direction dir.Direction,
	auroraAirport pkg.AuroraAirport,
	auroraFlight pkg.AuroraFlight,
) (*flight_status.FlightStatus, error) {
	var routePointFrom, routePointTo string
	if direction == dir.DEPARTURE {
		routePointFrom = auroraAirport.Airport
		routePointTo = auroraFlight.Destination
	} else {
		routePointFrom = auroraFlight.Destination
		routePointTo = auroraAirport.Airport
	}
	return &flight_status.FlightStatus{
		Airport:          auroraAirport.Airport,
		AirlineCode:      flightNumber.airline,
		FlightNumber:     flightNumber.number,
		FlightDate:       auroraFlight.Date,
		Direction:        auroraAirport.Direction,
		TimeActual:       parseAuroraActualTime(auroraFlight.Date, auroraFlight.TimeScheduled, auroraFlight.TimeActual),
		TimeScheduled:    auroraFlight.TimeScheduled,
		Status:           parseAuroraStatus(auroraFlight.Status, direction),
		Gate:             auroraFlight.GetGate(),
		Terminal:         auroraFlight.Terminal,
		CheckInDesks:     transformAuroraCheckinDesks(auroraFlight.CheckInDesks),
		BaggageCarousels: auroraFlight.BaggageCarousels,
		RoutePointFrom:   routePointFrom,
		RoutePointTo:     routePointTo,
		Source:           Aurora,
	}, nil
}

func parseAuroraStatus(status string, direction dir.Direction) string {
	if status == "" {
		return "no-data"
	}

	status = strings.ToLower(status)

	if statusIgnoreRegexp.MatchString(status) {
		return ""
	}

	for a, b := range statusConvertMap {
		if a == status {
			return b
		}
	}

	for a, b := range statusConvertMapByDirection[direction] {
		if a == status {
			return b
		}
	}

	for a, b := range statusConvertMapRegexp[direction] {
		if b.MatchString(status) {
			return a
		}
	}

	return "unknown"
}

var insideBrackets = regexp.MustCompile(`\([^\(\)]*\)`)

func transformAuroraCheckinDesks(rawDesks string) string {

	deskNumbers := make([]int, 0)
	rawDesks = strings.TrimSpace(insideBrackets.ReplaceAllString(rawDesks, ""))
	if len(rawDesks) == 0 || rawDesks == "-" {
		return ""
	}

	for _, desk := range strings.Split(rawDesks, ",") {
		desk = strings.TrimSpace(desk)
		if len(desk) == 0 {
			continue
		}

		deskNumber, err := strconv.Atoi(desk)
		if err != nil {
			intervalEnds := strings.Split(desk, "-")
			if len(intervalEnds) != 2 {
				logger.Logger().Warn("Cannot parse desk during parsing rawDesks",
					log.String("desk", desk),
					log.String("rawDesks", rawDesks),
				)
				continue
			}

			intervalStart, err := strconv.Atoi(intervalEnds[0])
			if err != nil {
				logger.Logger().Warn(
					"Bad interval end %s during parsing %s",
					log.String("intervalEnds[0]", intervalEnds[0]),
					log.String("desk", desk),
				)
			}

			intervalEnd, err := strconv.Atoi(intervalEnds[1])
			if err != nil {
				logger.Logger().Warn(
					"Bad interval end %s during parsing %s",
					log.String("intervalEnds[1]", intervalEnds[1]),
					log.String("desk", desk),
				)
			}

			if intervalEnd < intervalStart {
				intervalStart, intervalEnd = intervalEnd, intervalStart
			}

			// Unefficient, but it is a simple way to handle case, when there are neighbour intervals
			deskInIntervals := make([]int, intervalEnd-intervalStart+1)

			for i := intervalStart; i <= intervalEnd; i++ {
				deskInIntervals[i-intervalStart] = i
			}

			deskNumbers = append(deskNumbers, deskInIntervals...)

		} else {
			deskNumbers = append(deskNumbers, deskNumber)
		}
	}

	if len(deskNumbers) == 0 {
		logger.Logger().Warn("Can not parse any desk number", log.String("rawDesks", rawDesks))
		return rawDesks
	}

	sort.Ints(deskNumbers)

	var response []string
	intervalStart := deskNumbers[0]
	intervalEnd := deskNumbers[0]
	for i := 1; i < len(deskNumbers); i++ {
		currentDesk := deskNumbers[i]
		if currentDesk == intervalEnd+1 {
			intervalEnd++
		} else {
			response = append(response, writeInterval(intervalStart, intervalEnd))
			intervalStart = currentDesk
			intervalEnd = currentDesk
		}
	}

	response = append(response, writeInterval(intervalStart, intervalEnd))

	return strings.Join(response, ", ")
}

func writeInterval(intervalStart int, intervalEnd int) string {
	if intervalStart == intervalEnd {
		return strconv.Itoa(intervalStart)
	}

	return strconv.Itoa(intervalStart) + "-" + strconv.Itoa(intervalEnd)
}

func parseAuroraActualTime(date string, timeScheduled string, timeActual string) string {
	var t string
	var matches []string
	if timeActual != "" {
		t = timeActual
	} else if timeScheduled != "" {
		t = timeScheduled
	} else {
		return ""
	}

	matches = actualDatetimeRegexp.FindStringSubmatch(t)
	if len(matches) > 0 && matches[1] != "" {
		return matches[1]
	}

	matches = actualTimeRegexp.FindStringSubmatch(t)
	if len(matches) > 0 && matches[1] != "" {
		r := date + "T" + matches[1]
		if len(matches[1]) == 5 {
			r = r + ":00"
		}
		return r
	}

	return ""
}

func AuroraDecoder(bb []byte) (pkg.PartnerMessage, error) {
	var status AuroraMessage
	if err := json.Unmarshal(bb, &status.AuroraMessage); err != nil {
		var unmarshalTypeError *json.UnmarshalTypeError
		if xerrors.As(err, &unmarshalTypeError) {
			return nil, xerrors.Errorf(
				"AuroraDecoder input size %d, error at %d, near %v: %w",
				len(bb),
				unmarshalTypeError.Offset,
				string(aroundCharacter(bb, int(unmarshalTypeError.Offset), 32, 32)),
				err,
			)
		}
		return nil, xerrors.Errorf("AuroraDecoder: %w", err)
	}
	return status, nil
}

func aroundCharacter(bb []byte, pos, before, after int) []byte {
	maxPos := len(bb)
	if before < 0 {
		before = 0
	}
	if after < 0 {
		after = 0
	}
	from := pos - before
	if from < 0 {
		from = 0
	} else if from >= maxPos {
		from = maxPos
	}
	to := pos + after
	if to < 0 {
		to = 0
	} else if to >= maxPos {
		to = maxPos
	}
	return bb[from:to]
}
