package collectors

import (
	"encoding/json"

	"github.com/gofrs/uuid"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/flight_status_receiver/pkg"
	dir "a.yandex-team.ru/travel/avia/shared_flights/lib/go/direction"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/dtutil"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/strutil"
	"a.yandex-team.ru/travel/proto/avia/flight_status"
)

type FlightStatsAlert struct {
	pkg.FlightStatsMessage
}

func FlightStatsCollectorP() pkg.Collector {
	if c, err := DefaultCollector(
		FlightStats,
		pkg.DecoderFunc(FlightStatsDecoder),
	); err != nil {
		panic(err)
	} else {
		return c
	}

}

func (s FlightStatsAlert) Validate() error {
	if s.IsZero() {
		return xerrors.New("<FlightStatsAlert>.Validate: alert is empty")
	}
	return nil
}

func (s FlightStatsAlert) Normalize() ([]*flight_status.FlightStatus, error) {
	var statuses []*flight_status.FlightStatus

	fsStatus := s.Alert.FlightStatus

	airlineMapper := flightStatsAirlineCodeMapper(s.Appendix.Airlines.Airline)
	airportMapper := flightStatsAirportCodeMapper(s.Appendix.Airports.Airport)
	statuses = append(
		statuses,
		createStatusDTO(
			fsStatus.CarrierFsCode,
			fsStatus.FlightNumber,
			fsStatus,
			airlineMapper,
			airportMapper,
		)...,
	)

	for _, c := range fsStatus.Codeshares.Codeshares {
		statuses = append(
			statuses,
			createStatusDTO(
				c.CompanyIata,
				c.FlightNumber,
				fsStatus,
				airlineMapper,
				airportMapper,
			)...,
		)
	}

	for i := range statuses {
		statusUUID, err := uuid.NewV4()
		if err != nil {
			return nil, xerrors.Errorf("<FlightStatsAlert>.Normalize: %w", err)
		}
		statuses[i].StatusId = statusUUID.String()
	}

	return statuses, nil
}

func flightStatsAirportCodeMapper(airports []pkg.FlightStatsAirport) flightStatsCodeResolver {
	flightStatsToSomeCode := make(map[string]string)
	for _, airport := range airports {
		flightStatsToSomeCode[airport.Fs] = strutil.Coalesce(airport.Iata, airport.Icao)
	}
	return func(a string) string {
		if code, ok := flightStatsToSomeCode[a]; ok {
			return code
		}
		return a
	}
}

func flightStatsAirlineCodeMapper(airlines []pkg.FlightStatsAirline) flightStatsCodeResolver {
	flightStatsToSomeCode := make(map[string]string)
	for _, airline := range airlines {
		flightStatsToSomeCode[airline.Fs] = strutil.Coalesce(airline.Iata, airline.Icao)
	}
	return func(a string) string {
		if code, ok := flightStatsToSomeCode[a]; ok {
			return code
		}
		return a
	}
}

type flightStatsCodeResolver func(FsCode string) (IataOrSirenaCode string)

func createStatusDTO(
	flightStatsAirlineCode,
	flightNumber string,
	fsStatus pkg.FlightStatsStatus,
	airline flightStatsCodeResolver,
	airport flightStatsCodeResolver,
) []*flight_status.FlightStatus {

	var departureStatus = flight_status.FlightStatus{
		Airport:             airport(fsStatus.AirportFromFsCode),
		AirlineCode:         airline(flightStatsAirlineCode),
		FlightNumber:        flightNumber,
		FlightDate:          string(dtutil.FormatDateIso(fsStatus.GetDepartureDateTime())),
		Direction:           dir.DEPARTURE.String(),
		TimeActual:          dtutil.FormatDateTimeISO(fsStatus.GetDepartureActualDateTime()),
		TimeScheduled:       dtutil.FormatDateTimeISO(fsStatus.GetDepartureDateTime()),
		Status:              normalizeFlightStatsStatus(fsStatus.Status),
		Gate:                fsStatus.AirportResources.DepartureGate,
		Terminal:            fsStatus.AirportResources.DepartureTerminal,
		Diverted:            fsStatus.GetDiverted(),
		DivertedAirportCode: airport(fsStatus.DivertedAirportFsCode),
		RoutePointFrom:      airport(fsStatus.AirportFromFsCode),
		RoutePointTo:        airport(fsStatus.AirportToFsCode),
		Source:              FlightStats,
	}
	var arrivalStatus = flight_status.FlightStatus{
		Airport:             airport(fsStatus.AirportToFsCode),
		AirlineCode:         airline(flightStatsAirlineCode),
		FlightNumber:        flightNumber,
		FlightDate:          string(dtutil.FormatDateIso(fsStatus.GetArrivalDateTime())),
		Direction:           dir.ARRIVAL.String(),
		TimeActual:          dtutil.FormatDateTimeISO(fsStatus.GetArrivalActualDateTime()),
		TimeScheduled:       dtutil.FormatDateTimeISO(fsStatus.GetArrivalDateTime()),
		Status:              normalizeFlightStatsStatus(fsStatus.Status),
		Gate:                fsStatus.AirportResources.ArrivalGate,
		Terminal:            fsStatus.AirportResources.ArrivalTerminal,
		Diverted:            fsStatus.GetDiverted(),
		DivertedAirportCode: airport(fsStatus.DivertedAirportFsCode),
		RoutePointFrom:      airport(fsStatus.AirportFromFsCode),
		RoutePointTo:        airport(fsStatus.AirportToFsCode),
		Source:              FlightStats,
	}
	return []*flight_status.FlightStatus{&departureStatus, &arrivalStatus}
}

func normalizeFlightStatsStatus(s string) string {
	switch s {
	case "A":
		return "wait"
	case "C":
		return "cancelled"
	case "D":
		return "diverted"
	case "DN":
		return "unknown"
	case "L":
		return "arrived"
	case "NO":
		return "unknown"
	case "R":
		return "unknown"
	case "S":
		return "wait"
	case "U":
		return "unknown"
	default:
		return "no-data"
	}
}

func FlightStatsDecoder(bb []byte) (pkg.PartnerMessage, error) {
	status := FlightStatsAlert{}
	if err := json.Unmarshal(bb, &status.FlightStatsMessage); err != nil {
		return nil, xerrors.Errorf("FlightStatsDecoder: %v: %w", string(bb), err)
	}
	return status, nil
}
