package collectors

import (
	"encoding/json"

	"github.com/gofrs/uuid"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/flight_status_receiver/pkg"
	dir "a.yandex-team.ru/travel/avia/shared_flights/lib/go/direction"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
	"a.yandex-team.ru/travel/proto/avia/flight_status"
)

func OAGCollectorP() pkg.Collector {
	if c, err := DefaultCollector(
		OAG,
		pkg.DecoderFunc(OAGDecoder),
	); err != nil {
		panic(err)
	} else {
		return c
	}

}

type OAGMessage struct {
	pkg.OAGMessage
}

func (m OAGMessage) Validate() error {
	return nil
}

func (m OAGMessage) Normalize() ([]*flight_status.FlightStatus, error) {
	// TODO: нужно ли как-то обрабатывать recovery полет
	statuses, err := oagStatus(m.FlightData)
	if err != nil {
		return nil, xerrors.Errorf("<OAGMessage>.Normalize: %w", err)
	}

	for i := range statuses {
		statusUUID, err := uuid.NewV4()
		if err != nil {
			return nil, xerrors.Errorf("<OAGMessage>.Normalize: %w", err)
		}
		statuses[i].StatusId = statusUUID.String()
	}

	return statuses, nil
}

var oagStatusMap = map[string]string{
	"Unknown":       "unknown",
	"NoTakeoffInfo": "no-data",
	"Cancelled":     "cancelled",
	"Delayed":       "delay",
	"InAir":         "wait",
	"Landed":        "arrived",
	// TODO: какой статус имеют OAG статусы ниже?
	"OutGate":   "departed",
	"Scheduled": "wait",
	"InGate":    "arrived",
}

func parseOagStatus(oagStatus string) string {
	status, ok := oagStatusMap[oagStatus]
	if !ok {
		logger.Logger().Warn(
			"Failed to parse OAG status",
			log.String("oagStatus", oagStatus),
		)
		return "unknown"
	}
	return status
}

func oagStatus(flightData pkg.OAGFlightData) ([]*flight_status.FlightStatus, error) {
	var departureStatus = flight_status.FlightStatus{
		Airport:             flightData.DepartureAirportCode,
		AirlineCode:         flightData.AirlineCode,
		FlightNumber:        flightData.FlightNumber,
		FlightDate:          flightData.GetDepartureDate(),
		Direction:           dir.DEPARTURE.String(),
		TimeActual:          flightData.LatestDeparture.DateTimeLocal,
		TimeScheduled:       flightData.SchedDepartureLocal,
		Status:              parseOagStatus(flightData.Status),
		Gate:                flightData.DepartureGate,
		Terminal:            flightData.DepartureTerminal,
		CheckInDesks:        "",
		BaggageCarousels:    "",
		Diverted:            flightData.IsDiverted(),
		DivertedAirportCode: flightData.AlternateArrivalAirportCode,
		RoutePointFrom:      flightData.DepartureAirportCode,
		RoutePointTo:        flightData.ArrivalAirportCode,
		Source:              OAG,
	}
	var arrivalStatus = flight_status.FlightStatus{
		Airport:             flightData.ArrivalAirportCode,
		AirlineCode:         flightData.AirlineCode,
		FlightNumber:        flightData.FlightNumber,
		FlightDate:          flightData.GetArrivalDate(),
		Direction:           dir.ARRIVAL.String(),
		TimeActual:          flightData.LatestArrival.DateTimeLocal,
		TimeScheduled:       flightData.SchedArrivalLocal,
		Status:              parseOagStatus(flightData.Status),
		Gate:                flightData.ArrivalGate,
		Terminal:            flightData.ArrivalTerminal,
		CheckInDesks:        "",
		BaggageCarousels:    flightData.Baggage,
		Diverted:            flightData.IsDiverted(),
		DivertedAirportCode: flightData.AlternateArrivalAirportCode,
		RoutePointFrom:      flightData.DepartureAirportCode,
		RoutePointTo:        flightData.ArrivalAirportCode,
		Source:              OAG,
	}
	return []*flight_status.FlightStatus{&departureStatus, &arrivalStatus}, nil
}

func OAGDecoder(bb []byte) (pkg.PartnerMessage, error) {
	var status OAGMessage
	if err := json.Unmarshal(bb, &status.OAGMessage); err != nil {
		return nil, xerrors.Errorf("OAGDecoder: %v: %w", string(bb), err)
	}
	return status, nil
}
