package collectors

import (
	"encoding/json"
	"strings"

	"github.com/gofrs/uuid"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/flight_status_receiver/pkg"
	dir "a.yandex-team.ru/travel/avia/shared_flights/lib/go/direction"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
	"a.yandex-team.ru/travel/proto/avia/flight_status"
)

func VariFlightCollectorP() pkg.Collector {
	if c, err := DefaultCollector(
		VariFlight,
		pkg.DecoderFunc(VariFlightDecoder),
	); err != nil {
		panic(err)
	} else {
		return c
	}
}

type VariFlightMessage []pkg.VariFlightStatus

func (m VariFlightMessage) Validate() error {
	return nil
}

func (m VariFlightMessage) Normalize() ([]*flight_status.FlightStatus, error) {
	var flightStatuses []*flight_status.FlightStatus
	for i := range m {
		fs, err := variFlightStatus(m[i])
		if err != nil {
			return nil, xerrors.Errorf("<VariFlightMessage>.Normalize: %w", err)
		}
		for j := range fs {
			statusUUID, err := uuid.NewV4()
			if err != nil {
				return nil, xerrors.Errorf("<VariFlightMessage>.Normalize: %w", err)
			}
			fs[j].StatusId = statusUUID.String()

			flightStatuses = append(flightStatuses, fs[j])
		}
	}

	return flightStatuses, nil
}

var variFlightStatusMap = map[string]string{
	"schedule":  "wait",
	"delay":     "delay",
	"cancel":    "cancelled",
	"diversion": "diverted",
	"departure": "departed",
	"arrival":   "arrived",

	"return":                    "return",
	"diverted flight departure": "diverted flight departure",
	"diverted flight cancel":    "diverted flight cancel",
	"diverted flight arrival":   "diverted flight arrival/",
	"returned flight departure": "returned flight departure",
	"returned flight cancel":    "returned flight cancel",
	"returned flight arrival":   "returned flight arrival",
	"cancel in advance":         "cancel in advance",
	"circling":                  "circling",
	"circled":                   "circled",
	"climbing":                  "climbing",
	"cruising":                  "cruising",
	"descending":                "descending",
	"landing":                   "landing",
}

func parseVariFlightStatus(variFlightStatus string) string {
	status, ok := variFlightStatusMap[variFlightStatus]
	if !ok {
		logger.Logger().Warn(
			"Failed to parse VariFlight status",
			log.String("variFlightStatus", variFlightStatus),
		)
		return "unknown"
	}
	return status
}

func variFlightStatus(status pkg.VariFlightStatus) ([]*flight_status.FlightStatus, error) {
	var departureStatus = flight_status.FlightStatus{
		Airport:             status.DepartureAirportCode,
		AirlineCode:         status.FlightNumber[:2],
		FlightNumber:        status.FlightNumber[2:],
		FlightDate:          status.ScheduledDepartureDate(),
		Direction:           dir.DEPARTURE.String(),
		TimeActual:          strings.Replace(status.ActualDepartureTime(), " ", "T", 1),
		TimeScheduled:       strings.Replace(status.ScheduledDepartureDateTime, " ", "T", 1),
		Status:              parseVariFlightStatus(status.Status),
		Gate:                status.BoardGate,
		Terminal:            status.DepartureTerminal,
		CheckInDesks:        status.CheckInDesks,
		BaggageCarousels:    "",
		Diverted:            status.IsDiverted(),
		DivertedAirportCode: "",
		RoutePointFrom:      status.DepartureAirportCode,
		RoutePointTo:        status.ArrivalAirportCode,
		Source:              VariFlight,
	}
	if status.IsDiverted() {
		departureStatus.DivertedAirportCode = status.DivertedInfos[0].DepartureAirportCode
	}

	var arrivalStatus = flight_status.FlightStatus{
		Airport:             status.ArrivalAirportCode,
		AirlineCode:         status.FlightNumber[:2],
		FlightNumber:        status.FlightNumber[2:],
		FlightDate:          status.ScheduledArrivalDate(),
		Direction:           dir.ARRIVAL.String(),
		TimeActual:          strings.Replace(status.ActualArrivalTime(), " ", "T", 1),
		TimeScheduled:       strings.Replace(status.ScheduledArrivalDateTime, " ", "T", 1),
		Status:              parseVariFlightStatus(status.Status),
		Gate:                "",
		Terminal:            status.ArrivalTerminal,
		CheckInDesks:        "",
		BaggageCarousels:    status.BaggageCarousels,
		Diverted:            status.IsDiverted(),
		DivertedAirportCode: "",
		RoutePointFrom:      status.DepartureAirportCode,
		RoutePointTo:        status.ArrivalAirportCode,
		Source:              VariFlight,
	}
	if status.IsDiverted() {
		arrivalStatus.DivertedAirportCode = status.DivertedInfos[0].ArrivalAirportCode
	}

	return []*flight_status.FlightStatus{&departureStatus, &arrivalStatus}, nil
}

func VariFlightDecoder(bb []byte) (pkg.PartnerMessage, error) {
	var status VariFlightMessage
	if err := json.Unmarshal(bb, &status); err != nil {
		var unmarshalTypeError *json.UnmarshalTypeError
		if xerrors.As(err, &unmarshalTypeError) {
			return nil, xerrors.Errorf(
				"VariFlightDecoder input size %d, error at %d, near %v: %w",
				len(bb),
				unmarshalTypeError.Offset,
				string(aroundCharacter(bb, int(unmarshalTypeError.Offset), 32, 32)),
				err,
			)
		}
		return nil, xerrors.Errorf("VariFlightDecoder: %w", err)
	}

	return status, nil
}
