package delivery

import (
	"context"
	"fmt"
	"time"

	"github.com/cenkalti/backoff/v4"
	"golang.org/x/sync/errgroup"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/flight_status_receiver/pkg"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
)

type duplicator struct {
	deliveries []pkg.Delivery
}

func NewDuplicatorP(deliveries ...pkg.Delivery) pkg.Delivery {
	if d, err := NewDuplicator(deliveries...); err != nil {
		panic(err)
	} else {
		return d
	}
}

func NewDuplicator(deliveries ...pkg.Delivery) (pkg.Delivery, error) {
	if len(deliveries) == 0 {
		return nil, xerrors.Errorf("cannot create duplicator from nothing: %w", pkg.DeliveryIsNilError)
	}
	for _, d := range deliveries {
		if d == nil {
			return nil, xerrors.Errorf("cannot create duplicator: %w", pkg.DeliveryIsNilError)
		}
	}
	return duplicator{deliveries}, nil
}

func (d duplicator) String() string {
	return fmt.Sprintf("Duplicator: %+v", d.deliveries)
}

func (d duplicator) Deliver(ss interface{}, ctx context.Context) error {
	group, gctx := errgroup.WithContext(ctx)

	for _, delivery := range d.deliveries {
		func(d pkg.Delivery) {
			group.Go(func() error {
				return backoff.RetryNotify(
					func() error {
						return d.Deliver(ss, gctx)
					},
					backoff.WithMaxRetries(backoff.NewExponentialBackOff(), 3),
					func(err error, t time.Duration) {
						logger.Logger().Warn(
							"Delivery failed, retrying",
							log.Duration("delay", t),
							log.Error(err),
						)
					},
				)
			})
		}(delivery)

	}
	return group.Wait()
}
