package delivery

import (
	"context"
	"encoding/json"
	"sync"
	"sync/atomic"

	"github.com/cenkalti/backoff/v4"

	"a.yandex-team.ru/kikimr/public/sdk/go/persqueue"
	"a.yandex-team.ru/kikimr/public/sdk/go/persqueue/log/corelogadapter"
	"a.yandex-team.ru/kikimr/public/sdk/go/ydb"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/flight_status_receiver/pkg"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
)

type logbrokerStatusDelivery struct {
	lbWriter *logbrokerWriter
}

type WriteTopicWithCreds struct {
	Endpoint string
	Token    string
	Topic    string
	SourceID string
}

func NewLogbrokerDeliveryP(config WriteTopicWithCreds) pkg.Delivery {
	if d, err := NewLogbrokerDelivery(config); err != nil {
		panic(err)
	} else {
		return d
	}
}

func NewLogbrokerDelivery(config WriteTopicWithCreds) (pkg.Delivery, error) {
	if factory, err := newLogbrokerWriter(config); err != nil {
		return nil, xerrors.Errorf("NewLogbrokerDelivery: %w", err)
	} else {
		return &logbrokerStatusDelivery{factory}, nil
	}
}

func (d logbrokerStatusDelivery) String() string {
	return "Logbroker"
}
func (d logbrokerStatusDelivery) Deliver(ss interface{}, ctx context.Context) error {
	return backoff.Retry(
		func() error {
			return d.lbWriter.Write(ss, ctx)
		},
		backoff.WithMaxRetries(backoff.NewExponentialBackOff(), 5),
	)
}

type logbrokerWriter struct {
	writer persqueue.Writer
	seqNo  uint64
	sync.Mutex
}

func newLogbrokerWriter(config WriteTopicWithCreds) (*logbrokerWriter, error) {
	writer := persqueue.NewWriter(persqueue.WriterOptions{
		Credentials:    ydb.AuthTokenCredentials{AuthToken: config.Token},
		Endpoint:       config.Endpoint,
		Logger:         corelogadapter.New(logger.Logger()),
		RetryOnFailure: true,
		Topic:          config.Topic,
		SourceID:       []byte(config.SourceID),
		Codec:          persqueue.Raw,
	})
	ctx := context.Background()
	init, err := writer.Init(ctx)
	if err != nil {
		logger.Logger().Error("Unable to start writer", log.Error(err))
		return nil, xerrors.Errorf("newLogbrokerWriter: %w", err)
	}
	logger.Logger().Info("Initialized logbroker writer", log.Any("init", init))
	go func(p persqueue.Writer) (issues []*persqueue.Issue) {
		for rsp := range p.C() {
			switch m := rsp.(type) {
			case *persqueue.Ack:
				logger.Logger().Debug("Ack seqNo", log.UInt64("seq no", m.SeqNo))
			case *persqueue.Issue:
				logger.Logger().Error("Error", log.Error(m.Err))
				issues = append(issues, m)
			}
		}
		return
	}(writer)
	return &logbrokerWriter{writer: writer, seqNo: init.MaxSeqNo}, nil
}

func (d *logbrokerWriter) Write(ss interface{}, _ context.Context) error {
	d.Lock()
	defer d.Unlock()

	marshalled, err := json.Marshal(ss)
	if err != nil {
		logger.Logger().Fatal("Cannot marshal status", log.Error(err))
	}

	data := &persqueue.WriteMessage{
		Data: marshalled,
	}
	data.WithSeqNo(atomic.AddUint64(&d.seqNo, 1))
	if err = d.writer.Write(data); err != nil {
		logger.Logger().Fatal("Writer terminated", log.Error(err))
	}
	return nil
}

func (d *logbrokerWriter) Close() {
	d.Lock()
	defer d.Unlock()
	if err := d.writer.Close(); err != nil {
		logger.Logger().Fatal("Writer terminated unexpectedly", log.Error(err))
	}
}
