package handlers

import (
	"bytes"
	"encoding/json"
	"net/http"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/flight_status_receiver/pkg"
	"a.yandex-team.ru/travel/avia/library/go/handlers"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
)

func WriteJSON(w http.ResponseWriter, status int, body interface{}) error {
	w.WriteHeader(status)
	return json.NewEncoder(w).Encode(body)
}

type PipelineProvider interface {
	Pipeline(partner string) (pkg.Pipeline, error)
}

func DefaultHandler(manager PipelineProvider, pipelineName string) (handlers.Handler, error) {
	pipeLine, err := manager.Pipeline(pipelineName)
	if err != nil {
		return nil, xerrors.Errorf("DefaultHandler: %w", err)
	}
	handlerFunc := func(w http.ResponseWriter, r *http.Request) {
		var writerError error
		var handlerError error
		buf := bytes.Buffer{}
		_, handlerError = buf.ReadFrom(r.Body)
		if handlerError != nil {
			writerError = WriteJSON(w, http.StatusBadRequest, "error reading the request")
			logger.Logger().Error("Cannot read status", log.Error(handlerError))
			if writerError != nil {
				logger.Logger().Error("Error writing response", log.Error(writerError))
			}
			return
		}
		handlerError = pipeLine.Handle(buf.Bytes(), r.Context())
		if handlerError != nil {
			handlerStatus, handlerBody := HandleError(handlerError)
			writerError = WriteJSON(w, handlerStatus, handlerBody)
			logger.Logger().Error("Cannot handle status", log.Error(handlerError))
			if writerError != nil {
				logger.Logger().Error("Error writing response", log.Error(writerError))
			}
			return
		}
		writerError = WriteJSON(w, http.StatusOK, "OK")
		if writerError != nil {
			logger.Logger().Error("Error writing response", log.Error(writerError))
		}
	}

	return handlers.HandlerFunc{
		Handler: http.HandlerFunc(handlerFunc),
		Route:   "/" + pipelineName,
	}, nil
}

func HandleError(err error) (int, interface{}) {
	var httpError pkg.ErrorWithHTTPCode
	if xerrors.As(err, &httpError) {
		return httpError.HTTPCode(), httpError.Error()
	}
	return http.StatusInternalServerError, "cannot handle request"

}
