package pipeline

import (
	"context"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/flight_status_receiver/internal/metrics"
	"a.yandex-team.ru/travel/avia/flight_status_receiver/pkg"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
)

type defaultPipeline struct {
	pkg.Collector
	pkg.Delivery
	rawDataDelivery pkg.Delivery
}

func DefaultPipelineP(collector pkg.Collector, statusDelivery pkg.Delivery, rawDataDelivery pkg.Delivery) pkg.Pipeline {
	if p, err := DefaultPipeline(collector, statusDelivery, rawDataDelivery); err != nil {
		panic(err)
	} else {
		return p
	}
}

func DefaultPipeline(collector pkg.Collector, delivery pkg.Delivery, rawDataDelivery pkg.Delivery) (pkg.Pipeline, error) {
	if collector == nil {
		return nil, xerrors.Errorf("cannot create pipeline: %w", pkg.CollectorIsNilError)
	}
	if delivery == nil {
		return nil, xerrors.Errorf("cannot create pipeline: %w", pkg.DeliveryIsNilError)
	}
	return defaultPipeline{
		Collector:       collector,
		Delivery:        delivery,
		rawDataDelivery: rawDataDelivery,
	}, nil
}

func (p defaultPipeline) Handle(bb []byte, ctx context.Context) error {

	var err error
	errChan := make(chan error, 1)

	go func() {
		defer close(errChan)
		errChan <- func() error {
			var (
				ss             pkg.StatusPack
				collectorError error
			)
			defer func(tStart time.Time) {
				tEnd := time.Now()
				duration := tEnd.Sub(tStart)
				metrics.WritePipelineDuration(p.Name(), duration)
			}(time.Now())

			metrics.AddDataReceived(p.Name(), int64(len(bb)))
			ss, collectorError = p.Collect(bb, ctx)
			if collectorError != nil {
				metrics.IncrementPipelineFailure(p.Name())
				rawDeliveryError := p.rawDataDelivery.Deliver(RawData{
					StatusPack: ss,
					Partner:    p.Name(),
					Data:       string(bb),
					Error:      collectorError.Error(),
				}, ctx)
				if rawDeliveryError != nil {
					logger.Logger().Error("Cannot deliver invalid raw data", log.Any("collector", p.Name()))
				}
				return xerrors.Errorf("<defaultPipeline>.Handle: messageID: %v: %w", ss.MessageID, collectorError)
			}

			metrics.IncrementPipelineSuccess(p.Name())

			rawDeliveryError := p.rawDataDelivery.Deliver(RawData{
				StatusPack: ss,
				Partner:    p.Name(),
				Data:       string(bb),
			}, ctx)
			if rawDeliveryError != nil {
				logger.Logger().Error("Cannot deliver valid raw data", log.Any("collector", p.Name()))
			}
			return p.Deliver(ss.Statuses, ctx)
		}()
	}()

	select {
	case err = <-errChan:
		if err != nil {
			return xerrors.Errorf("handler failed: %w", err)
		}
		return nil
	case <-ctx.Done():
		return xerrors.Errorf("handler timed out: %w", pkg.TimeoutError)
	}
}

type RawData struct {
	pkg.StatusPack
	Partner string `json:"partner"`
	Data    string `json:"data"`
	Error   string `json:"error,omitempty"`
}
