package pkg

import (
	"time"

	"a.yandex-team.ru/library/go/core/log"
	dir "a.yandex-team.ru/travel/avia/shared_flights/lib/go/direction"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
)

const FlightStatsDatetimeFormat = "2006-01-02T15:04:05.000"

type FlightStatsMessage struct {
	Appendix FlightStatsAppendix `json:"appendix"`
	Alert    FlightStatsAlert    `json:"alert"`
}

func (fsm FlightStatsMessage) IsZero() bool {
	return fsm.Alert.IsZero()
}

type FlightStatsAlert struct {
	FlightStatus FlightStatsStatus `json:"flightStatus"`
}
type FlightStatsAirport struct {
	Name               string `json:"name"`
	ElevationFeet      string `json:"elevationFeet"`
	CountryCode        string `json:"countryCode"`
	Icao               string `json:"icao"`
	Latitude           string `json:"latitude"`
	Street1            string `json:"street1,omitempty"`
	LocalTime          string `json:"localTime"`
	CityCode           string `json:"cityCode"`
	UtcOffsetHours     string `json:"utcOffsetHours"`
	Iata               string `json:"iata"`
	Classification     string `json:"classification"`
	Fs                 string `json:"fs"`
	Faa                string `json:"faa"`
	Active             string `json:"active"`
	Longitude          string `json:"longitude"`
	RegionName         string `json:"regionName"`
	TimeZoneRegionName string `json:"timeZoneRegionName"`
	WeatherZone        string `json:"weatherZone"`
	City               string `json:"city"`
	CountryName        string `json:"countryName"`
}

type FlightStatsAirports struct {
	Airport []FlightStatsAirport `json:"airport"`
}

type FlightStatsAirline struct {
	Name        string `json:"name"`
	Icao        string `json:"icao"`
	Fs          string `json:"fs"`
	Active      string `json:"active"`
	Iata        string `json:"iata"`
	PhoneNumber string `json:"phoneNumber"`
}
type FlightStatsAirlines struct {
	Airline []FlightStatsAirline `json:"airline"`
}
type FlightStatsEquipment struct {
	Name      string `json:"name"`
	Regional  string `json:"regional"`
	Jet       string `json:"jet"`
	Iata      string `json:"iata"`
	Widebody  string `json:"widebody"`
	TurboProp string `json:"turboProp"`
}
type FlightStatsEquipments struct {
	Equipment []FlightStatsEquipment `json:"equipment"`
}
type FlightStatsAppendix struct {
	Airports   FlightStatsAirports   `json:"airports"`
	Airlines   FlightStatsAirlines   `json:"airlines"`
	Equipments FlightStatsEquipments `json:"equipments"`
}

func (fsa FlightStatsAlert) IsZero() bool {
	return fsa.FlightStatus.IsZero()
}

type FlightStatsStatus struct {
	CarrierFsCode         string                      `json:"carrierFsCode"`
	FlightNumber          string                      `json:"flightNumber"`
	Departure             FlighStatsDateTime          `json:"departureDate"`
	Arrival               FlighStatsDateTime          `json:"arrivalDate"`
	AirportFromFsCode     string                      `json:"departureAirportFsCode"`
	AirportToFsCode       string                      `json:"arrivalAirportFsCode"`
	Status                string                      `json:"status"`
	OperationalTimes      FlightStatsOperationalTimes `json:"operationalTimes"`
	Codeshares            FlightStatsCodeshares       `json:"codeshares"`
	AirportResources      FlightStatsAirportResources `json:"airportResources"`
	DivertedAirportFsCode string                      `json:"divertedAirportFsCode"`
}

func (d FlightStatsStatus) IsZero() bool {
	return d.CarrierFsCode == "" &&
		d.FlightNumber == "" &&
		d.Departure.IsZero() &&
		d.Arrival.IsZero() &&
		d.AirportFromFsCode == "" &&
		d.AirportToFsCode == "" &&
		d.Status == "" &&
		d.OperationalTimes.IsZero() &&
		d.Codeshares.IsZero() &&
		d.AirportResources.IsZero() &&
		d.DivertedAirportFsCode == ""
}

type FlightStatsOperationalTimes struct {
	ActualGateDeparture      FlighStatsDateTime `json:"actualGateDeparture"`
	EstimatedGateDeparture   FlighStatsDateTime `json:"estimatedGateDeparture"`
	ActualRunwayDeparture    FlighStatsDateTime `json:"actualRunwayDeparture"`
	EstimatedRunwayDeparture FlighStatsDateTime `json:"estimatedRunwayDeparture"`
	PublishedDeparture       FlighStatsDateTime `json:"publishedDeparture"`

	ActualGateArrival    FlighStatsDateTime `json:"actualGateArrival"`
	EstimatedGateArrival FlighStatsDateTime `json:"estimatedGateArrival"`
	PublishedArrival     FlighStatsDateTime `json:"publishedArrival"`
}

func (t FlightStatsOperationalTimes) IsZero() bool {
	return (FlightStatsOperationalTimes{}) == t
}

type FlighStatsDateTime struct {
	Local string `json:"dateLocal"`
	UTC   string `json:"dateUtc"`
}

func (t FlighStatsDateTime) IsZero() bool {
	return (FlighStatsDateTime{}) == t
}

type FlightStatsCodeshares struct {
	Codeshares []FlightStatsCodeshare `json:"codeshare"`
}

func (c FlightStatsCodeshares) IsZero() bool {
	return len(c.Codeshares) == 0
}

type FlightStatsCodeshare struct {
	CompanyIata  string `json:"fsCode"`
	FlightNumber string `json:"flightNumber"`
}

type FlightStatsAirportResources struct {
	DepartureTerminal string `json:"departureTerminal"`
	DepartureGate     string `json:"departureGate"`
	ArrivalTerminal   string `json:"arrivalTerminal"`
	ArrivalGate       string `json:"arrivalGate"`
}

func (r FlightStatsAirportResources) IsZero() bool {
	return (FlightStatsAirportResources{}) == r
}

type FlightStatsLog struct {
	Unixtime          int64       `json:"unixtime"`
	CompanyCode       string      `json:"company_code"`
	FlightNumber      string      `json:"flight_number"`
	FlightDate        string      `json:"flight_date"`
	AirportFromCode   string      `json:"airport_from_code"`
	AirportToCode     string      `json:"airport_to_code"`
	Departure         string      `json:"departure"`
	Arrival           string      `json:"arrival"`
	Status            string      `json:"status"`
	DepartureTerminal string      `json:"departure_terminal"`
	DepartureGate     string      `json:"departure_gate"`
	ArrivalTerminal   string      `json:"arrival_terminal"`
	ArrivalGate       string      `json:"arrival_gate"`
	Type              string      `json:"type"`
	Data              interface{} `json:"data"`

	DepartureDiverted            bool   `json:"departure_diverted"`
	DepartureDivertedAirportIata string `json:"departure_diverted_airport_iata"`
	ArrivalDiverted              bool   `json:"arrival_diverted"`
	ArrivalDivertedAirportIata   string `json:"arrival_diverted_airport_iata"`
}

func (d *FlightStatsStatus) GetDepartureDateTime() time.Time {
	t, err := time.Parse(FlightStatsDatetimeFormat, d.Departure.Local)
	if err != nil {
		logger.Logger().Warn(
			"Error parsing departure time (%s): %s",
			log.String("departure local", d.Departure.Local),
			log.Error(err),
		)
	}

	return t
}

func (d *FlightStatsStatus) GetArrivalDateTime() time.Time {
	t, err := time.Parse(FlightStatsDatetimeFormat, d.Arrival.Local)
	if err != nil {
		logger.Logger().Warn(
			"Error parsing arrival time (%s): %s",
			log.String("arrival local", d.Arrival.Local),
			log.Error(err),
		)
	}

	return t
}

func chooseTime(tt []string) time.Time {
	for _, t := range tt {
		if t == "" {
			continue
		}

		result, err := time.Parse(FlightStatsDatetimeFormat, t)
		if err != nil {
			logger.Logger().Warn("Error parsing time (%s): %s",
				log.String("time", t),
				log.Error(err),
			)
			continue
		}

		return result
	}

	return time.Time{}
}

func (d *FlightStatsStatus) GetDepartureActualDateTime() time.Time {
	return chooseTime([]string{
		d.OperationalTimes.ActualRunwayDeparture.Local,
		d.OperationalTimes.ActualGateDeparture.Local,
		d.OperationalTimes.EstimatedGateDeparture.Local,
		d.OperationalTimes.EstimatedRunwayDeparture.Local,
		d.OperationalTimes.PublishedDeparture.Local,
	})
}

func (d *FlightStatsStatus) GetArrivalActualDateTime() time.Time {
	return chooseTime([]string{
		d.OperationalTimes.ActualGateArrival.Local,
		d.OperationalTimes.EstimatedGateArrival.Local,
		d.OperationalTimes.PublishedArrival.Local,
	})
}

func (d *FlightStatsStatus) GetLog(direction dir.Direction) FlightStatsLog {
	l := FlightStatsLog{
		Unixtime:          time.Now().Unix(),
		CompanyCode:       d.CarrierFsCode,
		FlightNumber:      d.FlightNumber,
		AirportFromCode:   d.AirportFromFsCode,
		AirportToCode:     d.AirportToFsCode,
		Departure:         d.GetDepartureDateTime().Format("2006-01-02 15:04:05"),
		Arrival:           d.GetArrivalDateTime().Format("2006-01-02 15:04:05"),
		Status:            d.Status,
		DepartureTerminal: d.AirportResources.DepartureTerminal,
		DepartureGate:     d.AirportResources.DepartureGate,
		ArrivalTerminal:   d.AirportResources.ArrivalTerminal,
		ArrivalGate:       d.AirportResources.ArrivalGate,
	}

	switch direction {
	case dir.DEPARTURE:
		l.DepartureDiverted = d.GetDiverted()
		l.DepartureDivertedAirportIata = d.DivertedAirportFsCode
	case dir.ARRIVAL:
		l.ArrivalDiverted = d.GetDiverted()
		l.ArrivalDivertedAirportIata = d.DivertedAirportFsCode
	}
	return l
}

func (d *FlightStatsStatus) GetDiverted() bool {
	return d.Status == "D"
}
