package pkg

import (
	"context"
	"fmt"
	"time"

	"github.com/gofrs/uuid"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/proto/avia/flight_status"
)

type PipelineManager interface {
	RegisterPipeline(partner string, pipeline Pipeline) error
	Pipeline(partner string) (Pipeline, error)
}

type Pipeline interface {
	Handle([]byte, context.Context) error
}

type Collector interface {
	Name() string
	Collect(bb []byte, ctx context.Context) (StatusPack, error)
}

type StatusPack struct {
	MessageID  string                        `json:"message_id"`
	ReceivedAt int64                         `json:"received_at"`
	Statuses   []*flight_status.FlightStatus `json:"statuses"`
}

func (p *StatusPack) Init() error {
	UUID, err := uuid.NewV4()
	if err != nil {
		return xerrors.Errorf("<StatusPack>.Init: %w", err)
	}

	p.MessageID = UUID.String()
	p.ReceivedAt = time.Now().Unix()
	return nil
}

func (p *StatusPack) FillPartnerMessageInfo() {
	for i := range p.Statuses {
		p.Statuses[i].MessageId = p.MessageID
		p.Statuses[i].ReceivedAt = p.ReceivedAt
	}
}

type PartnerMessage interface {
	Validate() error
	Normalize() ([]*flight_status.FlightStatus, error)
}

type Decoder interface {
	Decode([]byte) (PartnerMessage, error)
}

type DecoderFunc func([]byte) (PartnerMessage, error)

func (d DecoderFunc) Decode(bb []byte) (PartnerMessage, error) {
	return d(bb)
}

type Delivery interface {
	fmt.Stringer
	Deliver(interface{}, context.Context) error
}
