from datetime import date  # noqa
from typing import Set, Optional, Type  # noqa

import ujson

from sandbox import sdk2
import sandbox.common.types.resource as ctr


def get_created_alerts(resource_type, departure_date, environment):
    # type: (Type[sdk2.Resource], date, str) -> Set[str]
    resource = _get_latest_created_alerts_resource_on_date(resource_type, departure_date, environment)
    if not resource:
        return set()
    with sdk2.ResourceData(resource).path.open() as f:
        return set(ujson.load(f))


def _get_latest_created_alerts_resource_on_date(resource_type, departure_date, environment):
    # type: (Type[sdk2.Resource], date, str) -> Optional[sdk2.Resource]
    return resource_type.find(
        attrs={'alerts_date': departure_date.strftime('%Y-%m-%d'), 'environment': environment},
        state=(ctr.State.READY,),
    ).first()


def _create_created_alerts_resource(resource_type, task, departure_date, environment):
    # type: (Type[sdk2.Resource], sdk2.Task, date, str) -> sdk2.Resource
    resource_file = 'fetch_flight_stats.created_alerts.json'
    return resource_type(
        task,
        'Created alerts for date: %s' % departure_date,
        path=resource_file,
        alerts_date=departure_date.strftime('%Y-%m-%d'),
        environment=environment,
    )


def save_created_alerts(resource_type, task, created_alerts, departure_date, environment):
    # type: (Type[sdk2.Resource], sdk2.Task, Set[str], date, str) -> None
    import ujson

    resource = _create_created_alerts_resource(resource_type, task, departure_date, environment)
    with sdk2.ResourceData(resource).path.open('wb') as f:
        ujson.dump(list(created_alerts), f)
    sdk2.ResourceData(resource).ready()
