# coding=utf-8
import logging

from requests import Session
from requests.packages.urllib3.util.retry import Retry
from requests.adapters import HTTPAdapter
from typing import Optional
from datetime import datetime, date

from travel.avia.flight_status_registrar.flight_stats.lib.alerts import save_created_alerts
from travel.avia.flight_status_registrar.lib.flights_fetcher import Flight
from travel.avia.flight_status_registrar.lib.registrar import AbstractRegistrar, RegistrarError
from travel.avia.library.python.iata_correction import IATACorrector

logger = logging.getLogger(__name__)


class FlightStatsRegistrationError(RegistrarError):
    pass


class FlightStatsRegistrar(AbstractRegistrar):
    DEPARTURE_TIMES_PRIORITY = (
        'actualRunwayDeparture',
        'actualGateDeparture',
        'estimatedGateDeparture',
        'estimatedRunwayDeparture',
        'publishedDeparture',
    )
    ARRIVAL_TIMES_PRIORITY = (
        'actualGateArrival',
        'estimatedGateArrival',
        'publishedArrival',
        # 'actualRunwayArrival',
        # 'estimatedRunwayArrival',
    )

    def __init__(
        self,
        api_url,
        app_id,
        app_key,
        shared_flights_api_url,
        http_connection_timeout,
        http_read_timeout,
        flight_stats_create_alert,
        created_alerts,
        environment,
        flight_stats_alert_handler,
        task=None,
        created_alerts_resource_type=None,
    ):
        self._shared_flights_api_url = shared_flights_api_url
        self._api_url = api_url
        self._app_id = app_id
        self._app_key = app_key
        self._http_connection_timeout = http_connection_timeout
        self._http_read_timeout = http_read_timeout
        self._flight_stats_create_alert = flight_stats_create_alert
        self._created_alerts = created_alerts
        self._environment = environment
        self._flight_stats_alert_handler = flight_stats_alert_handler
        self._task = task
        self._created_alerts_resource_type = created_alerts_resource_type

        self._session = Session()
        self._session.mount('http://', HTTPAdapter(
            max_retries=Retry(
                total=3,
                read=3,
                connect=3,
                backoff_factor=0.1,
                status_forcelist=(502,),
                method_whitelist=('GET',),
            ),
        ))

    def _fetch_flight_from_flight_stats(self, flight):
        # type: (Flight) -> Optional[dict]

        iata = flight.airline_code
        number = flight.number
        year, month, day = flight.departure_day.split('-')
        url = '{base_url}/flightstatus/rest/v2/json/flight/status/{iata}/{number}/dep/{year}/{month}/{day}'.format(
            base_url=self._api_url,
            iata=iata,
            number=number,
            year=year,
            month=month,
            day=day,
        )
        params = {
            'appId': self._app_id,
            'appKey': self._app_key,
            'utc': 'false',
            'codeType': 'IATA',
            'airport': flight.airport_from_code,
        }

        logging.info('Fetch url: %s with params %s', url, params)
        response = self._session.get(
            url,
            params=params,
            timeout=(self._http_connection_timeout, self._http_read_timeout),
        )
        response.raise_for_status()

        if response.status_code != 200:
            raise FlightStatsRegistrationError(
                'Error requesting flight stats: {} ({})'.format(response.status_code, response.reason),
            )

        data = response.json()
        error = data.get('error', None)
        if error:
            raise FlightStatsRegistrationError('Error requesting flight stats: {}'.format(error))

        flight_statuses = data.get('flightStatuses', [])
        if len(flight_statuses) == 0:
            raise FlightStatsRegistrationError(
                'Error requesting flight stats. Empty "flightStatuses": {}'.format(response.content),
            )

        appendix = data.get('appendix', [])
        if len(appendix) == 0:
            raise FlightStatsRegistrationError(
                'Error requesting flight stats. Empty "appendix": {}'.format(response.content),
            )

        return data

    def _get_time_actual(self, direction, status_data):
        # type: (str, dict) -> Optional[str]
        times = status_data.get('operationalTimes', None)
        if times is None:
            return None

        if direction == 'departure':
            for t in self.DEPARTURE_TIMES_PRIORITY:
                actual_time = times.get(t, None)
                if actual_time is None:
                    continue
                parsed_actual_time = self._parse_time(actual_time.get('dateLocal', None))
                if parsed_actual_time:
                    return parsed_actual_time
        elif direction == 'arrival':
            for t in self.ARRIVAL_TIMES_PRIORITY:
                actual_time = times.get(t, None)
                if actual_time is None:
                    continue
                parsed_actual_time = self._parse_time(actual_time.get('dateLocal', None))
                if parsed_actual_time:
                    return parsed_actual_time
        else:
            raise FlightStatsRegistrationError('Wrong direction "{}"'.format(direction))

    @staticmethod
    def _parse_time(s):
        # type: (str) -> Optional[str]
        if s is None:
            return None

        datetime_format = '%Y-%m-%dT%H:%M:%S.%f'

        try:
            t = datetime.strptime(s, datetime_format)
        except ValueError:
            logging.error('Couldn\'t parse string %s into datetime using format %s', s, datetime_format)
            t = None

        return t.strftime('%Y-%m-%dT%H:%M:%S') if t else None

    @staticmethod
    def _get_gate(direction, status_data):
        # type: (str, dict) -> Optional[str]
        airport_resources = status_data.get('airportResources', None)
        if airport_resources is None:
            return None

        if direction == 'departure':
            return airport_resources.get('departureGate')
        elif direction == 'arrival':
            return airport_resources.get('arrivalGate')
        else:
            raise RuntimeError('Wrong direction "{}"'.format(direction))

    @staticmethod
    def _get_terminal(direction, status_data):
        # type: (str, dict) -> Optional[str]
        airport_resources = status_data.get('airportResources', None)
        if airport_resources is None:
            return None

        if direction == 'departure':
            return airport_resources.get('departureTerminal')
        elif direction == 'arrival':
            return airport_resources.get('arrivalTerminal')
        else:
            raise RuntimeError('Wrong direction "{}"'.format(direction))

    def _get_codeshares(self, flight_statuses):
        # type: (dict) -> (int, str)
        codeshares = flight_statuses.get('codeshares', [])
        company_id_by_flight_number = IATACorrector(
            self._shared_flights_api_url,
        ).flight_numbers_to_carriers(
            ((c['fsCode'], c['flightNumber']) for c in codeshares),
        ) if codeshares else {}

        for c in codeshares:
            number = c['flightNumber']
            airline_id = self._get_airline_id(c['fsCode'], c['flightNumber'], company_id_by_flight_number)
            if airline_id:
                yield airline_id, number

    @staticmethod
    def _get_airline_id(code, number, company_id_by_flight_number):
        flight_number = u'{} {}'.format(code, number)
        c = company_id_by_flight_number.get(flight_number)
        if not c:
            logging.warning(u'Not found company by "{}"'.format(flight_number))
            return None

        return c

    def _build_alert_name(self, airline_code, flight_number, departure_date):
        # type: (str, str, date) -> str
        return '{}-{}{}-{}{}{}'.format(
            self._environment[:4],
            airline_code,
            flight_number,
            departure_date.year,
            departure_date.month,
            departure_date.day,
        )

    def _create_alert(
        self,
        alert_name,
        airline_code,
        flight_number,
        departure_date,
        airport_from_code,
        airport_to_code,
    ):
        # type: (str, str, str, date, str, str) -> Optional[dict]
        url = (
            '{base_url}/alerts/rest/v1/json/create/{iata}/{number}/from/{from_iata}/'
            'to/{to_iata}/departing/{year}/{month}/{day}'
        ).format(
            base_url=self._api_url,
            iata=airline_code,
            number=flight_number,
            year=departure_date.year,
            month=departure_date.month,
            day=departure_date.day,
            from_iata=airport_from_code,
            to_iata=airport_to_code,
        )
        params = {
            'appId': self._app_id,
            'appKey': self._app_key,
            'name': alert_name,
            'type': 'JSON',
            'deliverTo': self._flight_stats_alert_handler,
            'codeType': 'fs',
        }

        logging.info('Create alert. Fetch url: %s with params %s', url, params)
        response = self._session.get(
            url,
            params=params,
            timeout=(self._http_connection_timeout, self._http_read_timeout),
        )

        if response.status_code != 200:
            logging.error('Error create flight stats alert: %d (%s)', response.status_code, response.reason)
            return None

        data = response.json()
        error = data.get('error', None)
        if error:
            logging.error('Error create flight stats alert: %s', error)
            return None

        rule = data.get('rule')
        if not rule:
            logging.error('Empty rule: %s', data)
            return None

        logging.info('Created alert: %s', rule.get('id'))

        return data['rule']

    def register_flight(self, flight):
        # type: (Flight) -> bool
        fs_data = self._fetch_flight_from_flight_stats(flight)
        if not fs_data:
            return False

        if not self._flight_stats_create_alert:
            return True

        airline_code = next(iter(
            a['fs']
            for a in fs_data['appendix']['airlines']
            if a['fs'] == fs_data['request']['airline']['fsCode']
        ), None)
        if not airline_code:
            logging.info(
                'There is no appropriate airline for alert creation among %s. '
                'Requested airline fsCode: %s',
                fs_data['appendix']['airlines'],
                fs_data['request']['airline']['fsCode'],
            )
            return False
        flight_number = flight.number
        departure_date = datetime.strptime(flight.departure_day, '%Y-%m-%d').date()
        alert_name = self._build_alert_name(airline_code, flight_number, departure_date)
        alert = None
        is_alert_already_created = alert_name in self._created_alerts
        if not is_alert_already_created:
            alert = self._create_alert(
                alert_name,
                airline_code,
                flight_number,
                departure_date,
                flight.airport_from_code,
                flight.airport_to_code,
            )
        else:
            logging.info('Alert %s is already created', alert_name)

        if alert is None and not is_alert_already_created:
            return False

        self._created_alerts.add(alert_name)

        return True

    def save_created_alerts(self, departure_date):
        if not self._task:
            return
        save_created_alerts(
            self._created_alerts_resource_type,
            self._task,
            self._created_alerts,
            departure_date,
            self._environment,
        )
