# coding=utf-8
import logging
from abc import ABCMeta, abstractmethod

from typing import Iterator, List

from yt.wrapper import format


class Flight(object):
    def __init__(self, data):
        self.airline_code = data['airlineCode']
        self.number = data['number']
        self.airport_from_id = data['airportFromID']
        self.airport_from_code = data['airportFromCode']
        self.airport_to_id = data['airportToID']
        self.airport_to_code = data['airportToCode']
        self.departure_day = data['departureDay']
        self.departure_time = data['departureTime']
        self.arrival_day = data['arrivalDay']
        self.arrival_time = data['arrivalTime']
        operating_flight = data['operating']
        if operating_flight:
            title = operating_flight['title']
            parts = title.split(' ')
            if len(parts) != 2:
                raise Exception('Invalid flight title from shared-flights: {}'.format(title))
            self.airline_code = parts[0]
            self.number = parts[1]

    def key(self):
        return (self.airline_code, self.number)

    def __repr__(self):
        return '{}: {}'.format(type(self), self.__dict__)


class AbstractFlightsFetcher(object):
    __metaclass__ = ABCMeta

    @abstractmethod
    def get_flights(self, departure_date):
        pass


class PopularFlightsFetcher(AbstractFlightsFetcher):
    IGNORE_STATUS_SOURCE = ('aurora', 'airport')

    def __init__(
        self,
        yt_client,
        flights_fetcher,
        avia_tablo_fetcher,
        popular_flights_yt_table,
        flights_per_request=100,
    ):
        self._popular_flights_yt_table = popular_flights_yt_table
        self._flights_per_request = flights_per_request
        self._flights_fetcher = flights_fetcher
        self._avia_tablo_fetcher = avia_tablo_fetcher
        self._yt_client = yt_client

        self._airports_tablo_sources = None

    def get_flights(self, departure_date):
        departure_date_str = departure_date.strftime('%Y%m%d')
        request = set()
        requested_flights = set()

        for record in self._yt_client.read_table(self._popular_flights_yt_table, format.JsonFormat()):
            flight_number = record['flight_number'].replace(' ', '')
            flight_number = flight_number.strip()
            if not flight_number:
                continue
            request.add(departure_date_str + flight_number)
            if len(request) == self._flights_per_request:
                flights = self._flights_fetcher.flights(request)
                for f in self._iterate_flights(flights):
                    if f.key() in requested_flights:
                        continue
                    requested_flights.add(f.key())
                    yield f

                request.clear()

        if len(request) > 0:
            flights = self._flights_fetcher.flights(request)
            for f in self._iterate_flights(flights):
                if f.key() in requested_flights:
                    continue
                requested_flights.add(f.key())
                yield f

    def _iterate_flights(self, flights):
        # type: (List[dict]) -> Iterator[Flight]
        for flight in flights:
            for segment in self._construct_segments(flight):
                segment_number = u'{} {}'.format(segment.airline_code, segment.number)
                if not segment_number or not segment_number.strip():
                    logging.info('Skip %s because of empty segment number', flight)
                    continue
                if not segment.departure_day:
                    logging.info('Skip %s because of empty departure day', flight)
                    continue
                if not segment.arrival_day:
                    logging.info('Skip %s because of empty arrival day', flight)
                    continue
                fetch, reason = self._need_fetch_flight(segment)
                if fetch:
                    logging.info(u'Fetch "%s", %s', segment_number, reason)
                    yield segment
                else:
                    logging.info(u'Skip "%s", %s', segment_number, reason)

    @staticmethod
    def _construct_segments(flight):
        # type: (dict) -> Iterator[Flight]
        if flight.get('segments'):
            operating_flight = flight['operating']
            for segment in flight['segments']:
                if operating_flight:
                    segment['operating'] = operating_flight
                yield Flight(segment)
        else:
            yield Flight(flight)

    def _need_fetch_flight(self, flight):
        """
        Проверяем, нужно ли получать данные об этом рейсе.
        :param flight: рейс, в формате, получаемом от flight-storage
        :return: нужно ли получать данные об этом рейсе, причина получения / не получения (для логов)
        """
        if not self._airports_tablo_sources:
            self._airports_tablo_sources = self._avia_tablo_fetcher.get_airports_tablo_sources()

        reason = u'{airport_from_code} is from {source_from} and {airport_to_code} is from {source_to}'.format(
            airport_from_code=flight.airport_from_code,
            source_from=self._airports_tablo_sources.get(flight.airport_from_id),
            airport_to_code=flight.airport_to_code,
            source_to=self._airports_tablo_sources.get(flight.airport_to_id),
        )
        return (
            self._airports_tablo_sources.get(flight.airport_from_id) not in self.IGNORE_STATUS_SOURCE
            or self._airports_tablo_sources.get(flight.airport_to_id) not in self.IGNORE_STATUS_SOURCE
        ), reason
