# coding=utf-8
import logging

from urllib3 import Retry

import requests
from requests import Session
from requests.adapters import HTTPAdapter
from travel.avia.flight_status_registrar.lib.registrar import AbstractRegistrar, RegistrarError

BASE_URL = 'https://fnb.flightview.com'

logger = logging.getLogger(__name__)


class OAGRegistrationError(RegistrarError):
    pass


class OAGError(RegistrarError):
    def __init__(self, *args):
        super(OAGError, self).__init__(*args)
        self.departure_exception = None
        self.arrival_exception = None

    def add_error(self, direction, error):
        if direction == 'departure':
            self.departure_exception = error
        else:
            self.arrival_exception = error

    def need_raise(self):
        return (self.arrival_exception is not None
                or self.departure_exception is not None)

    def __str__(self):
        error = ''
        if self.departure_exception is not None:
            error += 'Departure error: %s ' % str(self.departure_exception)
        if self.arrival_exception is not None:
            error += 'Arrival error: %s ' % str(self.arrival_exception)
        return error


class OAGRegistrar(AbstractRegistrar):
    def __init__(self, app_id, auth_key):
        self._app_id = app_id
        self._auth_key = auth_key

        self._session = Session()
        adapter = HTTPAdapter(max_retries=Retry(
            total=3,
            read=3,
            connect=3,
            backoff_factor=0.1,
            status_forcelist=(500, 502, 503, 504),
        ))
        self._session.mount('http://', adapter)
        self._session.mount('https://', adapter)

    def register_flight(self, flight):
        error = OAGError()
        for direction, where in ('departure', 'from'), ('arrival', 'to'):
            year, month, day = getattr(flight, '%s_day' % direction).split('-')
            url = u'{base}/register/{direction}/{airport}/{carrier}/{flight_number}/{year}/{month}/{day}/{time}'.format(
                base=BASE_URL,
                direction=direction,
                airport=getattr(flight, 'airport_%s_code' % where),
                carrier=flight.airline_code,
                flight_number=flight.number,
                year=year, month=month, day=day,
                time=getattr(flight, '%s_time' % direction).replace(':', '')[:4],
            )
            logger.debug('OAG request: %s', url)
            r = self._session.post(url, json={'RequestParameters': {'appid': self._app_id, 'appkey': self._auth_key}})

            try:
                r.raise_for_status()
                response = r.json()

                if not response['Success']:
                    self._log_register_error(direction, flight, response['Error'])
                    error.add_error(direction, self._raise_error_by_body(response))
                else:
                    logger.info('%s flight %s %s (%s %s) has been registered in OAG' % (
                        direction.capitalize(),
                        flight.airline_code,
                        flight.number,
                        flight.departure_day,
                        flight.departure_time,
                    ))
            except (ValueError, KeyError) as e:
                self._log_register_error(direction, flight, 'Json error')
                # Считаем, что такой случай единичный и
                # мы можем продолжить обработку
                error.add_error(direction, OAGRegistrationError(e))
            except requests.HTTPError as e:
                self._log_register_error(direction, flight, 'Bad response status')
                error.add_error(direction, self._raise_error_by_status_code(e))

        if error.need_raise():
            raise error

    @staticmethod
    def _raise_error_by_body(response):
        # Считаем, что если начинается с Incorrect то ошибка
        # в запросе(неправильный запрос или неверные аутенфикационные данные),
        # поэтому не следует продолжать и кидаем фатал
        if response['Error'].startswith('Invalid'):
            raise RuntimeError(response['Error'])

        return OAGRegistrationError(response['Error'])

    @staticmethod
    def _raise_error_by_status_code(http_error):
        status_code = http_error.response.status_code

        # Считаем, что если коды ошибок как ниже, то ошибка
        # в запросе(неправильный запрос или неверные аутенфикационные данные),
        # поэтому не следует продолжать и кидаем фатал
        if status_code in (400, 401) or status_code >= 500:
            raise RuntimeError(http_error)

        return OAGRegistrationError(http_error)

    @staticmethod
    def _log_register_error(direction, flight, extra_msg):
        logger.error(
            'Error while trying register %s flight %s(%s %s). ' % (
                direction,
                flight.number,
                flight.departure_day,
                flight.departure_time,
            )
            + extra_msg
        )
