# coding=utf-8
import hashlib
import logging
from collections import OrderedDict

try:
    from urllib.parse import urlencode
except ImportError:
    from urllib import urlencode

from requests import Session
from requests.packages.urllib3.util.retry import Retry
from requests.adapters import HTTPAdapter


from travel.avia.flight_status_registrar.lib.flights_fetcher import Flight
from travel.avia.flight_status_registrar.lib.registrar import AbstractRegistrar, RegistrarError

logger = logging.getLogger(__name__)


class VariFlightRegistrationError(RegistrarError):
    pass


class VariFlightRegistrar(AbstractRegistrar):
    def __init__(self, api_uri, app_id, registration_code):
        self._app_id = app_id
        self._registration_code = registration_code
        self._api_uri = api_uri

        self._session = Session()
        self._session.mount('http://', HTTPAdapter(
            max_retries=Retry(
                total=3,
                read=3,
                connect=3,
                backoff_factor=0.1,
                status_forcelist=(502,),
                method_whitelist=('GET',),
            ),
        ))

    def register_flight(self, flight):
        # type: (Flight) -> None
        params = OrderedDict()
        params['appid'] = self._app_id
        params['arr'] = flight.airport_to_code
        params['date'] = flight.departure_day
        params['dep'] = flight.airport_from_code
        params['fnum'] = flight.airline_code + flight.number
        params['lang'] = 'en'
        logger.info('Params: %s', params)

        query = urlencode(params)
        url = '{api_url}/api/addflightpush/?{query}'.format(
            api_url=self._api_uri,
            query=query + '&token={}'.format(self._request_token(query)),
        )

        logger.info('variflight-request: %s', url)
        response = self._session.get(
            url=url,
        )
        response.raise_for_status()
        data = response.json()
        logger.info('variflight-response: %s', data)

        if data['error_code'] in (6, 8):
            return
        elif data['error_code'] in (5, 10):
            raise VariFlightRegistrationError(self._format_error(data['error_code'], data['error']))
        else:
            raise RuntimeError(self._format_error(data['error_code'], data['error']))

    def _request_token(self, query_string):
        encoded_request = hashlib.md5((query_string + self._registration_code).encode('utf-8')).hexdigest()
        return hashlib.md5(encoded_request.encode('utf-8')).hexdigest()

    @staticmethod
    def _format_error(error_code, error):
        return '{}-{}'.format(error_code, error)
