package logbroker

import (
	"context"
	"fmt"

	"a.yandex-team.ru/kikimr/public/sdk/go/persqueue"
)

type Reader struct {
	config          persqueue.ReaderOptions
	client          persqueue.Reader
	onBatchReceived func(persqueue.MessageBatch)
	onMessage       func(persqueue.ReadMessage)
}

type ReaderOption func(*Reader)

func WithOnBatchReceived(onBatchReceived func(persqueue.MessageBatch)) ReaderOption {
	return func(reader *Reader) {
		reader.onBatchReceived = onBatchReceived
	}
}

func WithOnMessage(onMessage func(persqueue.ReadMessage)) ReaderOption {
	return func(reader *Reader) {
		reader.onMessage = onMessage
	}
}

func NewReader(ctx context.Context, config persqueue.ReaderOptions, readerOptions ...ReaderOption) (*Reader, error) {
	client := persqueue.NewReader(config)

	if _, err := client.Start(ctx); err != nil {
		return nil, fmt.Errorf("unable to start reader: %w", err)
	}

	reader := &Reader{
		config:          config,
		client:          client,
		onBatchReceived: func(persqueue.MessageBatch) {},
		onMessage:       func(persqueue.ReadMessage) {},
	}
	for _, opt := range readerOptions {
		opt(reader)
	}
	return reader, nil
}

func (r *Reader) Read() error {
	for msg := range r.client.C() {
		switch v := msg.(type) {
		case *persqueue.Data:
			for _, b := range v.Batches() {
				r.onBatchReceived(b)
				for _, m := range b.Messages {
					r.onMessage(m)
				}
			}
			v.Commit()

		default:
		}
	}

	<-r.client.Closed()
	if err := r.client.Err(); err != nil {
		return fmt.Errorf("reader closed with error: %w", err)
	}
	return fmt.Errorf("reader closed without errors, but that should not have happened")
}
