package logging

import (
	"fmt"

	"go.uber.org/zap"
	"go.uber.org/zap/zapcore"

	"a.yandex-team.ru/library/go/core/log"
	arczap "a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/yandex/deploy/podagent"
	"a.yandex-team.ru/travel/library/go/logging"
)

func NewLogger(config *Config) (logger *arczap.Logger, err error) {
	var loggers []*arczap.Logger

	if config.needLogToStdout() {
		logger, err := getStdoutLogger(config)
		if err != nil {
			return nil, err
		}
		loggers = append(loggers, logger)
	}

	if config.needLogToFile() {
		var fileLogger *arczap.Logger
		if config.FileLoggingConfig.LogFormat == JSON {
			fileLogger, err = logging.NewYtFileLogger(config.Level, config.FileLoggingConfig.FilePath)
		} else {
			fileLogger, err = logging.NewFileLogger(config.Level, config.FileLoggingConfig.FilePath)
		}
		if err != nil {
			return nil, err
		}
		loggers = append(loggers, fileLogger)
	}

	if config.needLogToSentry() {
		sentry, err := logging.NewSentryLogger(log.ErrorLevel, config.SentryLoggingConfig.DSN)
		if err != nil {
			return nil, err
		}
		loggers = append(loggers, sentry)
	}

	var subCores []zapcore.Core
	for _, l := range loggers {
		subCores = append(subCores, l.L.Core())
	}
	core := zapcore.NewTee(subCores...)
	return arczap.NewWithCore(core, zap.AddCaller(), zap.AddStacktrace(zap.ErrorLevel)), nil
}

func getStdoutLogger(config *Config) (logger *arczap.Logger, err error) {
	if podagent.UnderPodAgent() {
		logger, err = arczap.NewDeployLogger(config.Level)
		if err != nil {
			return nil, fmt.Errorf("can't build deploy logger: %w", err)
		}
	} else {
		logger, err = arczap.NewQloudLogger(config.Level)
		if err != nil {
			return nil, fmt.Errorf("can't build qloud logger: %w", err)
		}
	}
	return logger, nil
}
