package metrics

import (
	libMetrics "a.yandex-team.ru/library/go/core/metrics"
	"a.yandex-team.ru/library/go/core/metrics/solomon"
)

type MetricsStorage struct {
	appMetricsRegistry *solomon.Registry
	registryByPrefix   map[MetricPrefix]libMetrics.Registry
	timerStore         map[MetricPrefix]map[MetricName]libMetrics.Timer
	timerVecStore      map[MetricPrefix]map[MetricName]libMetrics.TimerVec
	counterStore       map[MetricPrefix]map[MetricName]libMetrics.Counter
	counterVecStore    map[MetricPrefix]map[MetricName]libMetrics.CounterVec
	gaugeStore         map[MetricPrefix]map[MetricName]libMetrics.Gauge
}

func NewMetricsStorage(appMetricsRegistry *solomon.Registry) *MetricsStorage {
	return &MetricsStorage{
		appMetricsRegistry: appMetricsRegistry,
		registryByPrefix:   make(map[MetricPrefix]libMetrics.Registry),
		timerStore:         make(map[MetricPrefix]map[MetricName]libMetrics.Timer),
		timerVecStore:      make(map[MetricPrefix]map[MetricName]libMetrics.TimerVec),
		counterStore:       make(map[MetricPrefix]map[MetricName]libMetrics.Counter),
		counterVecStore:    make(map[MetricPrefix]map[MetricName]libMetrics.CounterVec),
		gaugeStore:         make(map[MetricPrefix]map[MetricName]libMetrics.Gauge),
	}
}

func createRatedHistogram(
	registry libMetrics.Registry, name MetricName, durationBuckets libMetrics.DurationBuckets) libMetrics.Timer {
	requestTimingsHistogram := registry.DurationHistogram(
		name.String(),
		durationBuckets,
	)
	solomon.Rated(requestTimingsHistogram)
	return requestTimingsHistogram
}

func createDurationHistogramVec(
	registry libMetrics.Registry, name MetricName, durationBuckets libMetrics.DurationBuckets, labels []string,
) libMetrics.TimerVec {
	durationHistogramVec := registry.DurationHistogramVec(
		name.String(),
		durationBuckets,
		labels,
	)
	return durationHistogramVec
}

func createRatedCounter(appMetricsRegistry libMetrics.Registry, name MetricName) libMetrics.Counter {
	counter := appMetricsRegistry.Counter(name.String())
	solomon.Rated(counter)
	return counter
}

func createCounter(appMetricsRegistry libMetrics.Registry, name MetricName) libMetrics.Counter {
	counter := appMetricsRegistry.Counter(name.String())
	return counter
}

func createCounterVec(appMetricsRegistry libMetrics.Registry, name MetricName, labels []string) libMetrics.CounterVec {
	counter := appMetricsRegistry.CounterVec(name.String(), labels)
	return counter
}

func createGauge(appMetricsRegistry libMetrics.Registry, name MetricName) libMetrics.Gauge {
	gauge := appMetricsRegistry.Gauge(name.String())
	return gauge
}

func createTimer(appMetricsRegistry libMetrics.Registry, name MetricName) libMetrics.Timer {
	timer := appMetricsRegistry.Timer(name.String())
	return timer
}

func createTimerVec(appMetricsRegistry libMetrics.Registry, name MetricName, labels []string) libMetrics.TimerVec {
	timer := appMetricsRegistry.TimerVec(name.String(), labels)
	return timer
}

func (wm *MetricsStorage) RatedCounter(prefix MetricPrefix, name MetricName) libMetrics.Counter {
	if _, found := wm.counterStore[prefix]; !found {
		wm.counterStore[prefix] = make(map[MetricName]libMetrics.Counter)
	}
	if counter, found := wm.counterStore[prefix][name]; found {
		return counter
	}
	if _, found := wm.registryByPrefix[prefix]; !found {
		wm.registryByPrefix[prefix] = wm.appMetricsRegistry.WithPrefix(prefix.String())
	}
	counter := createRatedCounter(wm.registryByPrefix[prefix], name)
	wm.counterStore[prefix][name] = counter
	return counter
}

func (wm *MetricsStorage) Counter(prefix MetricPrefix, name MetricName) libMetrics.Counter {
	if _, found := wm.counterStore[prefix]; !found {
		wm.counterStore[prefix] = make(map[MetricName]libMetrics.Counter)
	}
	if counter, found := wm.counterStore[prefix][name]; found {
		return counter
	}
	if _, found := wm.registryByPrefix[prefix]; !found {
		wm.registryByPrefix[prefix] = wm.appMetricsRegistry.WithPrefix(prefix.String())
	}
	counter := createCounter(wm.registryByPrefix[prefix], name)
	wm.counterStore[prefix][name] = counter
	return counter
}

func (wm *MetricsStorage) CounterVec(prefix MetricPrefix, name MetricName, labels []string) libMetrics.CounterVec {
	if _, found := wm.counterVecStore[prefix]; !found {
		wm.counterVecStore[prefix] = make(map[MetricName]libMetrics.CounterVec)
	}
	if counter, found := wm.counterVecStore[prefix][name]; found {
		return counter
	}
	if _, found := wm.registryByPrefix[prefix]; !found {
		wm.registryByPrefix[prefix] = wm.appMetricsRegistry.WithPrefix(prefix.String())
	}
	counter := createCounterVec(wm.registryByPrefix[prefix], name, labels)
	wm.counterVecStore[prefix][name] = counter
	return counter
}

func (wm *MetricsStorage) Gauge(prefix MetricPrefix, name MetricName) libMetrics.Gauge {
	if _, found := wm.gaugeStore[prefix]; !found {
		wm.gaugeStore[prefix] = make(map[MetricName]libMetrics.Gauge)
	}
	if counter, found := wm.gaugeStore[prefix][name]; found {
		return counter
	}
	if _, found := wm.registryByPrefix[prefix]; !found {
		wm.registryByPrefix[prefix] = wm.appMetricsRegistry.WithPrefix(prefix.String())
	}
	gauge := createGauge(wm.registryByPrefix[prefix], name)
	wm.gaugeStore[prefix][name] = gauge
	return gauge
}

func (wm *MetricsStorage) Timer(prefix MetricPrefix, name MetricName) libMetrics.Timer {
	if _, found := wm.timerStore[prefix]; !found {
		wm.timerStore[prefix] = make(map[MetricName]libMetrics.Timer)
	}
	if timer, found := wm.timerStore[prefix][name]; found {
		return timer
	}
	if _, found := wm.registryByPrefix[prefix]; !found {
		wm.registryByPrefix[prefix] = wm.appMetricsRegistry.WithPrefix(prefix.String())
	}
	timer := createTimer(wm.registryByPrefix[prefix], name)
	wm.timerStore[prefix][name] = timer
	return timer
}

func (wm *MetricsStorage) TimerVec(prefix MetricPrefix, name MetricName, labels []string) libMetrics.TimerVec {
	if _, found := wm.timerVecStore[prefix]; !found {
		wm.timerVecStore[prefix] = make(map[MetricName]libMetrics.TimerVec)
	}
	if timer, found := wm.timerVecStore[prefix][name]; found {
		return timer
	}
	if _, found := wm.registryByPrefix[prefix]; !found {
		wm.registryByPrefix[prefix] = wm.appMetricsRegistry.WithPrefix(prefix.String())
	}
	timer := createTimerVec(wm.registryByPrefix[prefix], name, labels)
	wm.timerVecStore[prefix][name] = timer
	return timer
}

func (wm *MetricsStorage) RatedHistogram(
	prefix MetricPrefix, name MetricName, buckets libMetrics.DurationBuckets) libMetrics.Timer {
	if _, found := wm.timerStore[prefix]; !found {
		wm.timerStore[prefix] = make(map[MetricName]libMetrics.Timer)
	}
	if timer, found := wm.timerStore[prefix][name]; found {
		return timer
	}
	if _, found := wm.registryByPrefix[prefix]; !found {
		wm.registryByPrefix[prefix] = wm.appMetricsRegistry.WithPrefix(prefix.String())
	}
	timer := createRatedHistogram(wm.registryByPrefix[prefix], name, buckets)
	wm.timerStore[prefix][name] = timer
	return timer
}

func (wm *MetricsStorage) RatedHistogramVec(
	prefix MetricPrefix, name MetricName, buckets libMetrics.DurationBuckets, labels []string,
) libMetrics.TimerVec {
	if _, found := wm.timerVecStore[prefix]; !found {
		wm.timerVecStore[prefix] = make(map[MetricName]libMetrics.TimerVec)
	}
	if timer, found := wm.timerVecStore[prefix][name]; found {
		return timer
	}
	if _, found := wm.registryByPrefix[prefix]; !found {
		wm.registryByPrefix[prefix] = wm.appMetricsRegistry.WithPrefix(prefix.String())
	}
	timer := createDurationHistogramVec(wm.registryByPrefix[prefix], name, buckets, labels)
	wm.timerVecStore[prefix][name] = timer
	return timer
}

type MetricPrefix string
type MetricName string

func (m MetricPrefix) String() string {
	return string(m)
}

func (m MetricName) String() string {
	return string(m)
}

var globalMetrics *MetricsStorage

func SetGlobalMetrics(metrics *MetricsStorage) {
	globalMetrics = metrics
}

func GlobalMetrics() *MetricsStorage {
	return globalMetrics
}
