package probes

import (
	"fmt"
	"time"

	"a.yandex-team.ru/library/go/core/log"
)

type State struct {
	logger            log.Logger
	stopped           bool
	onReadyHooks      []func() error
	onStopBeforeHooks []func()
	onStopAfterHooks  []func()
}

func NewState(logger log.Logger, options ...StateOption) *State {
	s := &State{logger: logger}
	for _, opt := range options {
		opt(s)
	}
	return s
}

func (s State) Ready() error {
	if s.stopped {
		return fmt.Errorf("service stopped")
	}

	for _, hook := range s.onReadyHooks {
		if err := hook(); err != nil {
			s.logger.Error("service unavailable", log.Error(err))
			return err
		}
	}
	return nil
}

func (s *State) Stop(delay time.Duration) {
	s.logger.Infof("Stopping was called with delay=%s", delay)
	s.invokeHooks(s.onStopBeforeHooks)
	s.stop(delay)
	s.invokeHooks(s.onStopAfterHooks)
}

func (s *State) stop(delay time.Duration) {
	s.stopped = true
	time.Sleep(delay)
}

func (s *State) invokeHooks(hooks []func()) {
	defer func() {
		if r := recover(); r != nil {
			s.logger.Errorf("An error is caught while invoke hooks: %+v", r)
		}
	}()
	for _, hook := range hooks {
		hook()
	}
}
