package searchcontext

import (
	"fmt"
	"strconv"
	"strings"
	"time"

	timeformats "cuelang.org/go/pkg/time"

	"a.yandex-team.ru/library/go/core/xerrors"
)

func ParseQID(rawQID string) (QID, error) {
	qidParts := strings.Split(rawQID, ".")
	if len(qidParts) != 5 {
		return QID{}, xerrors.Errorf("invalid QID format: %s", rawQID)
	}
	rawCreatedAt := qidParts[0]
	service := qidParts[1]
	tCode := qidParts[2]
	rawQueryKey := qidParts[3]
	lang := qidParts[4]

	queryKey, err := ParseQKey(rawQueryKey)
	if err != nil {
		return QID{}, err
	}
	createdAt, err := parseCreatedAt(rawCreatedAt)
	if err != nil {
		return QID{}, err
	}
	return QID{
		CreatedAt: createdAt,
		Service:   service,
		TCode:     tCode,
		QKey:      queryKey,
		Lang:      lang,
		QID:       rawQID,
	}, nil
}

func ParseQKey(rawQueryKey string) (QKey, error) {
	parts := strings.Split(rawQueryKey, "_")
	if len(parts) != 9 {
		return QKey{}, xerrors.Errorf("invalid QKey format: %s", rawQueryKey)
	}
	dateForward, err := parseDate(parts[2])
	if err != nil {
		return QKey{}, err
	}
	dateBackward, err := parseDate(parts[3])
	if err != nil {
		return QKey{}, err
	}
	adults, err := parsePassengers(parts[5])
	if err != nil {
		return QKey{}, err
	}
	children, err := parsePassengers(parts[6])
	if err != nil {
		return QKey{}, err
	}
	infants, err := parsePassengers(parts[7])
	if err != nil {
		return QKey{}, err
	}
	return QKey{
		PointFromKey:    parts[0],
		PointToKey:      parts[1],
		DateForward:     dateForward,
		DateBackward:    dateBackward,
		Class:           parts[4],
		Adults:          adults,
		Children:        children,
		Infants:         infants,
		NationalVersion: parts[8],
	}, nil
}

func ExtractRawQKeyFromQID(rawQID string) (string, error) {
	qidParts := strings.Split(rawQID, ".")
	if len(qidParts) != 5 {
		return "", xerrors.Errorf("invalid QID format: %s", rawQID)
	}
	return qidParts[3], nil
}

func parseCreatedAt(rawCreatedAt string) (time.Time, error) {
	parts := strings.Split(rawCreatedAt, "-")
	return time.Parse("060102T150405.000", fmt.Sprintf("%sT%s.%s", parts[0], parts[1], parts[2]))
}

func parsePassengers(rawPassengers string) (uint8, error) {
	const base = 10
	const size = 8
	parsed, err := strconv.ParseUint(rawPassengers, base, size)
	if err != nil {
		return 0, err
	}
	return uint8(parsed), nil
}

func parseDate(rawDate string) (time.Time, error) {
	if rawDate == "None" {
		return time.Time{}, nil
	}
	return time.Parse(timeformats.RFC3339Date, rawDate)
}
