package backend

import (
	"bytes"
	"encoding/json"
	"fmt"
	"net/http"
	"net/url"
	"time"

	"github.com/cenkalti/backoff/v4"

	"a.yandex-team.ru/library/go/httputil/headers"
	"a.yandex-team.ru/travel/avia/library/go/services/backend/partners"
)

type Client interface {
	GetPartners() ([]*partners.WizardPartner, error)
}

type HTTPClient struct {
	backendBaseURL string
	httpClient     *http.Client
}

type methodName string

const (
	partnersMethodName methodName = "wizardPartners"
)

func NewHTTPClient(backendBaseURL string, httpClient *http.Client) *HTTPClient {
	return &HTTPClient{backendBaseURL: backendBaseURL, httpClient: httpClient}
}

func (client *HTTPClient) GetPartners() (partners []*partners.WizardPartner, err error) {
	request := func() error {
		partners, err = client.partnersRequest()
		return err
	}

	requestBackoff := &backoff.ExponentialBackOff{
		InitialInterval:     backoff.DefaultInitialInterval,
		RandomizationFactor: backoff.DefaultRandomizationFactor,
		Multiplier:          backoff.DefaultMultiplier,
		MaxInterval:         time.Second,
		MaxElapsedTime:      3 * time.Second,
		Clock:               backoff.SystemClock,
		Stop:                backoff.Stop,
	}
	requestBackoff.Reset()

	err = backoff.Retry(request, requestBackoff)
	if err != nil {
		return nil, err
	}
	return partners, nil
}

func (client *HTTPClient) partnersRequest() (partners []*partners.WizardPartner, err error) {
	requestURL, err := client.buildURL(partnersMethodName)
	if err != nil {
		return nil, err
	}
	body := []map[string]string{{"name": string(partnersMethodName)}}

	jsonBody, _ := json.Marshal(body)
	bodyBytes := bytes.NewBuffer(jsonBody)

	response, err := client.httpClient.Post(requestURL, headers.TypeApplicationJSON.String(), bodyBytes)
	if err != nil {
		return nil, fmt.Errorf("an error occured while getting partners: %+v", err)
	}
	defer response.Body.Close()
	if response.StatusCode != http.StatusOK {
		return nil, fmt.Errorf("avia-backend has responded with %+v status code", response.StatusCode)
	}
	partnersResponse := WizardPartnersResponse{}
	err = json.NewDecoder(response.Body).Decode(&partnersResponse)
	if err != nil {
		return nil, fmt.Errorf("couldn't decode backend partners response: %+v", err)
	}
	if partnersResponse.Status == ErrorResponseStatus {
		return nil, fmt.Errorf("backend answered with an error: %s while getting partners", partnersResponse.Reason)
	}
	return partnersResponse.Data[0], err
}

func (client *HTTPClient) buildURL(methodName methodName) (requestURL string, err error) {
	baseURL, err := url.Parse(client.backendBaseURL)
	if err != nil {
		return requestURL, fmt.Errorf("bad backend base url: %+v. An error occured while parsing: %+v", client.backendBaseURL, err)
	}
	query := baseURL.Query()
	query.Set("name", string(methodName))
	baseURL.RawQuery = query.Encode()
	return baseURL.String(), nil
}
