package featureflag

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"time"

	"github.com/cenkalti/backoff/v4"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/library/go/containers"
)

var retryableStatusCodes = containers.SetOf(http.StatusInternalServerError, http.StatusBadGateway, http.StatusGatewayTimeout)

type Client interface {
	CreateFlags() (flags Flags, err error)
}

type HTTPClient struct {
	host        string
	serviceCode string
	logger      log.Logger
	httpClient  *http.Client
}

func NewClient(host string, serviceCode string, logger log.Logger, httpClient *http.Client) *HTTPClient {
	return &HTTPClient{
		host:        host,
		serviceCode: serviceCode,
		logger:      logger,
		httpClient:  httpClient,
	}
}

func (client *HTTPClient) CreateFlags() (flags Flags, err error) {
	requestURL := fmt.Sprintf("%s/feature-flag?service-code=%s", client.host, client.serviceCode)
	requestBackoff := &backoff.ExponentialBackOff{
		InitialInterval:     backoff.DefaultInitialInterval,
		RandomizationFactor: backoff.DefaultRandomizationFactor,
		Multiplier:          backoff.DefaultMultiplier,
		MaxInterval:         time.Second,
		MaxElapsedTime:      3 * time.Second,
		Clock:               backoff.SystemClock,
		Stop:                backoff.Stop,
	}
	requestBackoff.Reset()

	request := func() error {
		response, err := client.httpClient.Get(requestURL)
		if err != nil {
			return err
		}
		if retryableStatusCodes.Contains(response.StatusCode) {
			body, err := ioutil.ReadAll(response.Body)
			message := fmt.Sprintf("temporary error occured during request feature-flag-api. status code: %d.", response.StatusCode)
			if err != nil {
				message += fmt.Sprintf(" body: %s", body)
			}
			return fmt.Errorf(message)
		}
		featureFlagAPIResponse := APIResponse{}
		err = json.NewDecoder(response.Body).Decode(&featureFlagAPIResponse)
		if err != nil {
			return err
		}
		if featureFlagAPIResponse.ABFlags == nil {
			flags = NewFlags(featureFlagAPIResponse.Flags, containers.SetOf[string]())
		} else {
			flags = NewFlags(featureFlagAPIResponse.Flags, *featureFlagAPIResponse.ABFlags)
		}
		return nil
	}
	err = backoff.Retry(request, requestBackoff)
	if err != nil {
		client.logger.Error(fmt.Sprintf("error %+v occured during request %s", err, requestURL))
	}
	return flags, err
}

type APIResponse struct {
	Flags   containers.Set[string]  `json:"flags"`
	ABFlags *containers.Set[string] `json:"abFlags"`
}
