package featureflag

import (
	"fmt"
	"sync/atomic"
	"time"

	"github.com/jonboulle/clockwork"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/library/go/containers"
)

type StorageInterface interface {
	GetFlags() *Flags
	StartPeriodicUpdates()
	StopPeriodicUpdates()
}

type Storage struct {
	featureFlagAPIClient   Client
	flagsUpdateInterval    time.Duration
	flags                  atomic.Value
	logger                 log.Logger
	enablePeriodicUpdating atomic.Value
	clock                  clockwork.Clock
}

func NewStorage(
	featureFlagAPIClient Client,
	flagsUpdateInterval time.Duration,
	logger log.Logger,
	clock clockwork.Clock,
) *Storage {
	storage := Storage{
		featureFlagAPIClient: featureFlagAPIClient,
		flagsUpdateInterval:  flagsUpdateInterval,
		logger:               logger,
		clock:                clock,
	}
	err := storage.updateFlags()
	if err != nil {
		flags := NewFlags(
			containers.SetOf[string](),
			containers.SetOf[string](),
		)
		storage.flags.Store(&flags)
	}
	return &storage
}

func (storage *Storage) GetFlags() *Flags {
	return storage.flags.Load().(*Flags)
}

func (storage *Storage) updateFlags() error {
	flags, err := storage.featureFlagAPIClient.CreateFlags()

	if err != nil {
		return err
	}
	storage.flags.Store(&flags)

	return nil
}

func (storage *Storage) updateFlagsPeriodically() {
	ticker := storage.clock.NewTicker(storage.flagsUpdateInterval)
	for range ticker.Chan() {
		if !storage.enablePeriodicUpdating.Load().(bool) {
			return
		}
		err := storage.updateFlags()
		if err != nil {
			storage.logger.Error(fmt.Sprintf("error occured during updating feature flag storage flags: %+v", err))
		}
	}
}

func (storage *Storage) StartPeriodicUpdates() {
	storage.enablePeriodicUpdating.Store(true)
	go storage.updateFlagsPeriodically()
}

func (storage *Storage) StopPeriodicUpdates() {
	storage.enablePeriodicUpdating.Store(false)
}

type MockStorage struct {
	flags *Flags
}

func NewMockStorage(
	flags *Flags,
) *MockStorage {
	return &MockStorage{
		flags: flags,
	}
}

func (storage *MockStorage) GetFlags() *Flags {
	return storage.flags
}

func (storage *MockStorage) StartPeriodicUpdates() {}

func (storage *MockStorage) StopPeriodicUpdates() {}
