package ru.yandex.avia.booking.partners.gateways.aeroflot;

import java.io.IOException;
import java.net.ConnectException;
import java.net.UnknownHostException;
import java.time.Clock;
import java.time.Instant;
import java.time.ZoneId;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeoutException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableMap;
import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.avia.booking.model.SearchRequest;
import ru.yandex.avia.booking.partners.gateways.BookingGateway;
import ru.yandex.avia.booking.partners.gateways.BookingRetryableException;
import ru.yandex.avia.booking.partners.gateways.aeroflot.AeroflotProviderProperties.WhiteMonday2020Promo;
import ru.yandex.avia.booking.partners.gateways.aeroflot.converter.AeroflotVariantConverter;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotOrderCreateResult;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotOrderRef;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotOrderStatus;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotOrderSubStatus;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotRequestContext;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotServicePayload;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotTicketCoupon;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotTicketDocTypeCode;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotTotalOffer;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotVariant;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.SearchData;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.AeroflotNdcApiV3CompatibilityConverter;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.AeroflotNdcApiV3Helper;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.AeroflotNdcApiV3VariantsSynchronizer;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.AirShoppingRq;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.AirShoppingRs;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.ApiDataXmlRoot;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.Offer;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.OfferPriceRq;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.OfferPriceRs;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.OrderCreateRq;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.OrderRetrieveRq;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.OrderViewRs;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.OrderViewRsBody;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.PaymentCard;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.PaymentInstructions;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.StatusMessage;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.TicketDocInfo;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.requests.AeroflotNdcApiV3ApiResponseException;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.requests.AeroflotNdcApiV3ModelXmlConverter;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.requests.AeroflotNdcApiV3ModelXmlConverterConfig;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.requests.AeroflotNdcApiV3RequestFactory;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.requests.AeroflotNdcApiV3RequestFactoryConfig;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.requests.OfferPriceRequestParams;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.requests.OrderCreateRequestParams;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.requests.OrderRetrieveRequestParams;
import ru.yandex.avia.booking.partners.gateways.model.availability.AvailabilityCheckRequest;
import ru.yandex.avia.booking.partners.gateways.model.availability.AvailabilityCheckResponse;
import ru.yandex.avia.booking.partners.gateways.model.availability.DepartureIsTooCloseException;
import ru.yandex.avia.booking.partners.gateways.model.availability.PriceChangedException;
import ru.yandex.avia.booking.partners.gateways.model.availability.VariantNotAvailableException;
import ru.yandex.avia.booking.partners.gateways.model.booking.BookingFailureException;
import ru.yandex.avia.booking.partners.gateways.model.booking.BookingFailureReason;
import ru.yandex.avia.booking.partners.gateways.model.booking.ClientInfo;
import ru.yandex.avia.booking.partners.gateways.model.booking.ServicePayload;
import ru.yandex.avia.booking.partners.gateways.model.booking.ServicePayloadInitParams;
import ru.yandex.avia.booking.partners.gateways.model.booking.TravellerInfo;
import ru.yandex.avia.booking.partners.gateways.model.payment.PaymentFailureReason;
import ru.yandex.avia.booking.partners.gateways.model.search.Variant;
import ru.yandex.avia.booking.partners.gateways.utils.TransliterationUtils;
import ru.yandex.avia.booking.remote.RpcContext;
import ru.yandex.avia.booking.remote.http.HttpClient;
import ru.yandex.avia.booking.services.tdapi.AviaTicketDaemonUtils;
import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper;

import static java.util.Comparator.comparing;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;

@Slf4j
public class AeroflotGateway implements BookingGateway {
    public static final String AEROFLOT_PARTNER_ID = "aeroflot";
    private static final ZoneId PNR_DATE_TIME_ZONE_ID = ZoneId.of("Europe/Moscow");
    private static final Pattern CUSTOM_ERROR_MESSAGE_PATTERN = Pattern.compile("<!--([\\w]+Exception): ([^>]+)-->");

    private final AeroflotProviderProperties config;
    private final AeroflotDynamicPropertiesProvider dynamicConfig;
    private final HttpClient httpClient;

    private final AeroflotNdcApiV3RequestFactory requestFactory;
    private final AeroflotNdcApiV3ModelXmlConverter xmlConverter;
    private final AeroflotNdcApiV3CompatibilityConverter v3CompatibilityConverter;

    public AeroflotGateway(@NonNull AeroflotProviderProperties config,
                           @NonNull AsyncHttpClientWrapper aeroflotAhc,
                           @NonNull Clock clock) {
        log.info("Aeroflot client: booking_url={}, user_name={}, clock={}",
                config.getBookingUrl(), config.getUserName(), clock.instant());
        this.config = config;
        this.dynamicConfig = buildDynamicConfig(config, clock);
        this.httpClient = new HttpClient(() -> ImmutableMap.of(
                "X-IBM-Client-Id", dynamicConfig.getAuthToken().getValue(),
                "Content-Type", config.getContentType()),
                aeroflotAhc);

        // todo(tlg-13): TRAVELBACK-1149: should be moved to the config
        this.requestFactory = new AeroflotNdcApiV3RequestFactory(
                AeroflotNdcApiV3RequestFactoryConfig.builder()
                        .aggregatorId(() -> dynamicConfig.getAggregatorId().getValue())
                        .apiVersion("18.2")
                        .build());
        this.xmlConverter = new AeroflotNdcApiV3ModelXmlConverter(
                AeroflotNdcApiV3ModelXmlConverterConfig.builder()
                        .unknownPropertiesAllowed(true)
                        .prettyPrinterEnabled(false)
                        .build());
        this.v3CompatibilityConverter = new AeroflotNdcApiV3CompatibilityConverter();
    }

    private AeroflotDynamicPropertiesProvider buildDynamicConfig(AeroflotProviderProperties config, Clock clock) {
        AeroflotDynamicPropertiesProvider provider = new AeroflotDynamicPropertiesProvider(clock);
        provider.getAggregatorId().setValue(config.getUserName());
        provider.getAuthToken().setValue(config.getAuthToken());
        WhiteMonday2020Promo promo2020 = config.getPromo2020();
        if (promo2020 != null && promo2020.getEnabled() == Boolean.TRUE) {
            Preconditions.checkArgument(promo2020.getStartsAt().isBefore(promo2020.getEndsAt()),
                    "Illegal startsAt & endsAt combination: [%s, %s)", promo2020.getStartsAt(), promo2020.getEndsAt());
            // enabling the promo properties at startsAt
            provider.getAggregatorId().setValue(promo2020.getUserName(), promo2020.getStartsAt());
            provider.getAuthToken().setValue(promo2020.getAuthToken(), promo2020.getStartsAt());
            // disabling the promo properties (restoring the initial ones) at endsAt
            provider.getAggregatorId().setValue(config.getUserName(), promo2020.getEndsAt());
            provider.getAuthToken().setValue(config.getAuthToken(), promo2020.getEndsAt());
        }
        return provider;
    }

    public AirShoppingRs searchVariants(SearchRequest request) {
        AirShoppingRq airShoppingRq = requestFactory.createAirShoppingRq(request, true);
        String responseXml;
        try {
            responseXml = sendHttpRequest(false, xmlConverter.convertToXml(airShoppingRq), RpcContext.empty());
        } catch (Exception e) {
            if (e instanceof InterruptedException) {
                // preserving the status
                Thread.currentThread().interrupt();
            }
            throw new RuntimeException(e);
        }
        AirShoppingRs airShoppingRs = convertFromXmlAndValidate(responseXml, AirShoppingRs.class);
        if (airShoppingRs.getError() != null) {
            throw new RuntimeException("AirShoppingRQ failed: " + airShoppingRs.getError());
        }
        return airShoppingRs;
    }

    @Override
    public AeroflotVariant resolveVariantInfo(@NonNull JsonNode dataNode) {
        return resolveVariantInfoImpl(dataNode, false);
    }

    @Override
    public AeroflotVariant resolveVariantInfoAndOptimizeJson(@NonNull JsonNode dataNode) {
        return resolveVariantInfoImpl(dataNode, true);
    }

    private AeroflotVariant resolveVariantInfoImpl(@NonNull JsonNode dataNode, boolean trimData) {
        JsonNode orderData = dataNode.path("order_data");
        Preconditions.checkArgument(orderData.get("booking_info") instanceof ObjectNode, "booking_info not found");
        ObjectNode bookingInfo = (ObjectNode) orderData.get("booking_info");

        String offerId = extractNotEmptyText(bookingInfo, "OfferId");
        SearchData searchData = SearchData.builder()
                .qid(orderData.path("qid").textValue())
                .externalBookingUrl(orderData.path("url").textValue())
                .searchParams(AviaTicketDaemonUtils.parseSearchParamsSafe(dataNode))
                .variantTag(dataNode.path("variant").path("tag").textValue())
                .build();
        if (Strings.isNullOrEmpty(searchData.getQid()) || Strings.isNullOrEmpty(searchData.getExternalBookingUrl())) {
            log.warn("Incomplete search data: qid={}, externalBookingUrl={}", searchData.getQid(),
                    searchData.getExternalBookingUrl());
        }

        String airShoppingXml = extractNotEmptyText(bookingInfo, "AirShoppingRS");
        AirShoppingRs airShoppingRs = convertFromXmlAndValidate(airShoppingXml, AirShoppingRs.class);
        airShoppingRs = requestFactory.createTrimmedAirShoppingRsData(airShoppingRs, offerId);
        if (trimData) {
            // preventing the huge useless value from being saved into DB
            String trimmedXml = xmlConverter.convertToXml(airShoppingRs);
            bookingInfo.put("AirShoppingRS", trimmedXml);
        }

        AeroflotRequestContext context = AeroflotRequestContext.builder()
                .cabinType(Integer.valueOf(extractNotEmptyText(bookingInfo, "CabinType")))
                .countryCode(extractNotEmptyText(bookingInfo, "CountryCode"))
                .language(extractNotEmptyText(bookingInfo, "LanguageCode"))
                .responseId(airShoppingRs.getResponse().getShoppingResponse().getShoppingResponseID())
                .build();
        return v3CompatibilityConverter.convertToV1Variant(airShoppingRs, offerId, context, searchData);
    }

    @Override
    public void synchronizeUpdatedVariantInfoJson(JsonNode dataNode, Variant variantInfo) {
        ObjectNode bookingInfo = (ObjectNode) dataNode.at("/order_data/booking_info");
        String offerId = extractNotEmptyText(bookingInfo, "OfferId");
        String airShoppingXml = extractNotEmptyText(bookingInfo, "AirShoppingRS");
        AirShoppingRs airShoppingRs = convertFromXmlAndValidate(airShoppingXml, AirShoppingRs.class);
        airShoppingRs = requestFactory.createTrimmedAirShoppingRsData(airShoppingRs, offerId);

        AirShoppingRs updatedAirShoppingRs = AeroflotNdcApiV3VariantsSynchronizer
                .synchronizeVariants(airShoppingRs, variantInfo);
        if (updatedAirShoppingRs != null) {
            log.info("The variant info has been updated, storing a new version");
            bookingInfo.put("AirShoppingRS", xmlConverter.convertToXml(updatedAirShoppingRs));
        }
    }

    @Override
    public String getExternalVariantId(Object variantInfo) {
        AeroflotVariant variant = (AeroflotVariant) variantInfo;
        return variant.getOffer().getId();
    }

    @Override
    public AeroflotServicePayload createServicePayload(@NonNull ServicePayloadInitParams params) {
        AeroflotVariant variant = (AeroflotVariant) params.getVariantToken();
        AeroflotServicePayload payload = new AeroflotServicePayload();
        payload.setVariantId(params.getVariantId());
        payload.setVariant(variant);
        payload.setClientInfo(params.getClientInfo());
        payload.setPreliminaryCost(params.getPreliminaryPrice());
        payload.setPartnerId(AEROFLOT_PARTNER_ID);
        payload.setTravellers(convertTravellers(params.getTravellers()));
        payload.setFareTerms(params.getFareTerms());
        payload.setPromoCampaignsInfo(params.getPromoCampaignsInfo());
        return payload;
    }

    @Override
    public Class<? extends ServicePayload> getPayloadType() {
        return AeroflotServicePayload.class;
    }

    @Override
    public AvailabilityCheckResponse checkAvailabilityAll(@NonNull AvailabilityCheckRequest request) {
        try {
            AeroflotVariant variant = (AeroflotVariant) request.getToken();
            AeroflotVariant updated = checkAvailabilityImpl(variant, true).get();
            Variant genericVariant = AeroflotVariantConverter.convertVariant(updated);
            return AvailabilityCheckResponse.builder()
                    .variant(genericVariant)
                    .build();
        } catch (InterruptedException | ExecutionException e) {
            throw convertOfferPriceCompletionException(e);
        }
    }

    @NonNull
    private CompletableFuture<AeroflotVariant> checkAvailabilityImpl(
            AeroflotVariant variant, boolean checkAll) {
        Map<String, CompletableFuture<OfferPriceRs>> checks = new LinkedHashMap<>();
        AirShoppingRs airShoppingRs = v3CompatibilityConverter.convertToV3AirShoppingDataForPriceCheck(variant);
        List<Offer> offersToCheck = checkAll ?
                airShoppingRs.getResponse().getOffersGroup().getCarrierOffers().getOffer() :
                List.of(AeroflotNdcApiV3Helper.findOfferById(airShoppingRs, variant.getOffer().getId()));
        for (Offer offer : offersToCheck) {
            OfferPriceRq offerPriceRq = requestFactory.createOfferPriceRq(OfferPriceRequestParams.builder()
                    .language(variant.getContext().getLanguage())
                    .countryOfSale(variant.getContext().getCountryCode())
                    .shoppingResponseId(variant.getContext().getResponseId())
                    .dataLists(airShoppingRs.getResponse().getDataLists())
                    .offer(offer)
                    .build());
            String xmlReq = xmlConverter.convertToXml(offerPriceRq);
            checks.put(offer.getOfferID(), sendAsyncHttpRequest(xmlReq, RpcContext.builder()
                    // todo(tlg-13): TRAVELBACK-1149: fall back to defaults only when there are no values
                    .extraHeaders(ImmutableMap.of(
                            "X-Forwarded-For", "127.0.0.1",
                            "User-Agent", config.getUserAgent()
                    ))
                    .build())
                    .thenApply(
                            responseAndStatus -> convertFromXmlAndValidate(
                                    responseAndStatus.response(), OfferPriceRs.class)));
        }
        return CompletableFuture.allOf(checks.values().toArray(new CompletableFuture[0]))
                .handle((r, t) -> {
                    Map<String, AeroflotVariant> variants = new LinkedHashMap<>();
                    Map<String, Throwable> errors = new HashMap<>();
                    for (Map.Entry<String, CompletableFuture<OfferPriceRs>> entry : checks.entrySet()) {
                        String offerId = entry.getKey();
                        try {
                            OfferPriceRs response = entry.getValue().join();
                            handleOfferPriceErrors(response.getError(), offerId);

                            AeroflotVariant checkedVariant = v3CompatibilityConverter
                                    .convertToV1Variant(response, variant)
                                    .setSearchData(variant.getSearchData())
                                    .setContext(variant.getContext());
                            checkedVariant.getOffer().setId(offerId);
                            variants.put(offerId, checkedVariant);
                        } catch (CompletionException e) {
                            errors.put(offerId, convertOfferPriceCompletionException(e));
                        } catch (Exception e) {
                            errors.put(offerId, e);
                        }
                    }
                    if (!errors.isEmpty()) {
                        log.info("Availability check errors: {}", errors);
                    }
                    String mainOfferId = variant.getOffer().getId();
                    if (variants.isEmpty()) {
                        throw combineOfferPriceTariffExceptions(errors.values());
                    }
                    AeroflotVariant mainVariant = variants.get(mainOfferId);
                    if (mainVariant == null) {
                        // the variants collection is never empty if we reach this block
                        mainVariant = variants.values().stream()
                                .min(comparing(v -> v.getOffer().getTotalPrice()))
                                .orElseThrow(() -> new IllegalStateException("Should never happen"));
                    }
                    mainVariant.setAllTariffs(variant.getAllTariffs().stream()
                            .map(ao -> variants.get(ao.getId()))
                            .filter(Objects::nonNull)
                            .map(AeroflotVariant::getOffer)
                            .sorted(comparing(AeroflotTotalOffer::getTotalPrice))
                            .collect(toList()));
                    return mainVariant;
                });
    }

    private List<TravellerInfo> convertTravellers(List<TravellerInfo> travellers) {
        return travellers.stream()
                .map(t -> t.toBuilder()
                        .documentNumber(TransliterationUtils.transliterateToLatinSafeIcao9303(t.getDocumentNumber()))
                        .build())
                .collect(toList());
    }

    @NonNull
    public AeroflotVariant checkAvailabilitySingle(@NonNull AeroflotVariant variant) {
        try {
            return checkAvailabilityImpl(variant, false).get();
        } catch (InterruptedException | ExecutionException e) {
            throw convertOfferPriceCompletionException(e);
        }
    }

    private RuntimeException convertOfferPriceCompletionException(Exception e) {
        Throwable t = e instanceof ExecutionException || e instanceof CompletionException ? e.getCause() : e;
        if (t instanceof IOException || t instanceof TimeoutException || t instanceof InterruptedException
                || t instanceof BookingRetryableException) {
            if (t instanceof InterruptedException) {
                // preserving the status
                Thread.currentThread().interrupt();
            }
            return new BookingRetryableException("Recoverable error during OfferPriceRQ; " +
                    "e.msg=" + t.getMessage(), t);
        }
        // wrap even runtime exceptions to build reasonable stack traces
        if (t instanceof DepartureIsTooCloseException) {
            return new DepartureIsTooCloseException(e);
        }
        if (t instanceof VariantNotAvailableException) {
            return new VariantNotAvailableException(e);
        }
        Throwable exceptionToRethrow = t != null ? t : e;
        return new RuntimeException("Unexpected OfferPriceRQ exception; " +
                "e.msg=" + exceptionToRethrow.getMessage(), exceptionToRethrow);
    }

    private RuntimeException combineOfferPriceTariffExceptions(Collection<Throwable> errors) {
        // keep the original message if there are no other failed checks
        String message = errors.size() == 1 ?
                errors.iterator().next().getMessage() :
                "None of the " + errors.size() + " tariffs is available at the moment";
        // in any case we wrap the exceptions from the futures to get properly filled stack traces
        RuntimeException resultEx;
        if (errors.stream().allMatch(e -> e instanceof DepartureIsTooCloseException)) {
            resultEx = new DepartureIsTooCloseException(message);
        } else if (errors.stream().allMatch(e -> e instanceof VariantNotAvailableException)) {
            resultEx = new VariantNotAvailableException(message);
        } else if (errors.stream().allMatch(e -> e instanceof BookingRetryableException)) {
            resultEx = new BookingRetryableException(message);
        } else {
            resultEx = new RuntimeException(message);
        }
        for (Throwable err : errors) {
            resultEx.addSuppressed(err);
        }
        return resultEx;
    }

    private void handleOfferPriceErrors(StatusMessage error, @NonNull String offerId) {
        if (error == null) {
            return;
        }

        String description = error.describe();
        if (!Strings.isNullOrEmpty(error.getCode())) {
            switch (AeroflotNdcApiErrorCode.getCode(error.getCode(), error.getDescText())) {
                case TEMPORARILY_UNAVAILABLE:
                    throw new BookingRetryableException("NDC API is temporarily unavailable: " + description);
                case UNABLE_TO_PROCESS:
                    switch (AeroflotNdcApiCustomErrorCode.getByValue(error.getCustomType())) {
                        case SABRE_NO_FARE_FOR_CLASS:
                            throw new VariantNotAvailableException("The tariff isn't available anymore: " + description);
                        case SABRE_NO_SUCH_FLIGHT:
                            throw new VariantNotAvailableException("The flight isn't available anymore: " + description);
                        case DEPARTURE_IS_TOO_CLOSE:
                            // special case of N/A exception for external redirects on availability checks in API
                            throw new DepartureIsTooCloseException("The offer isn't available anymore: " + description);
                        default:
                            throw new BookingRetryableException("NDC API has failed but it could be a temporary " +
                                    "problem: " + description);
                    }
                case SEGMENT_DEPARTURE_IS_TOO_CLOSE:
                    // special case of N/A exception for external redirects on availability checks in API
                    throw new DepartureIsTooCloseException("The offer isn't available anymore: " + description);
                case NOT_AVAILABLE:
                case SABRE_FLIGHT_NO_OP:
                case INVALID_DEPARTURE_DATE:
                    throw new VariantNotAvailableException("The offer isn't available anymore: offer_id=" + offerId + ", " + description);
                case TARIFF_NOT_AVAILABLE:
                case FARE_NOT_AVAILABLE:
                    throw new VariantNotAvailableException("The tariff isn't available anymore: offer_id=" + offerId + ", " + description);
            }
        }

        throw new RuntimeException(String.format(
                "Unexpected OfferPriceRQ API error for offer id %s: %s", offerId, error.describe()));
    }

/*
    private void handleCommonErrors(Document rsp) {
        // <errorResponse><httpCode>429</httpCode><httpMessage>Too Many Requests</httpMessage><moreInformation>Rate
        Limit exceeded</moreInformation></errorResponse>
        if ("429".equals(rsp.valueOf("/errorResponse/httpCode"))) {
            throw new BookingTooManyRequestsException(rsp.asXML());
        }
    }
*/

    @NonNull
    private String sendHttpRequest(
            boolean isBookingRequest, @NonNull String xmlRequest, @NonNull RpcContext context)
            throws IOException, TimeoutException, InterruptedException {
        return getHttpResponseWithStatus(isBookingRequest, xmlRequest, context).response();
    }

    @NonNull
    private HttpClient.ResponseWithStatus getHttpResponseWithStatus(
            boolean isBookingRequest, @NonNull String xmlRequest, @NonNull RpcContext context)
            throws IOException, TimeoutException, InterruptedException {
        String apiUrl = isBookingRequest ? config.getBookingUrl() : config.getSearchUrl();
        return httpClient.sendRequest(apiUrl, xmlRequest, config.getReadTimeout(), context);
    }

    @NonNull
    private CompletableFuture<HttpClient.ResponseWithStatus> sendAsyncHttpRequest(
            @NonNull String xmlRequest, @NonNull RpcContext context) {
        return httpClient.sendAsyncRequest(config.getBookingUrl(), xmlRequest, config.getReadTimeout(), context);
    }

    private <T extends ApiDataXmlRoot<T>> T convertFromXmlAndValidate(String xml, Class<T> responseClass) {
        T result = xmlConverter.convertFromXml(xml, responseClass);

        // can be replaced with another mechanism when it's available:
        // https://github.com/FasterXML/jackson-dataformat-xml/issues/414
        // (Add optional verification of root name matching)
        if (!result.getExpectedXmlns().equals(result.getXmlns())) {
            throw AeroflotNdcApiV3ApiResponseException.createTruncated(xml,
                    String.format("Schema mismatch: expected=%s, actual=%s",
                            result.getExpectedXmlns(), result.getXmlns()));
        }

        // some special errors are passed as comments which can't be parsed by the used library
        return tryAppendErrorComments(xml, result);
    }

    private <T extends ApiDataXmlRoot<T>> T tryAppendErrorComments(String xml, T apiResponse) {
        StatusMessage error = apiResponse.getGenericError();
        if (error == null) {
            return apiResponse;
        }
        Matcher matcher = CUSTOM_ERROR_MESSAGE_PATTERN.matcher(xml);
        if (matcher.find()) {
            String customErrorType = xml.substring(matcher.start(1), matcher.end(1));
            String customErrorComment = xml.substring(matcher.start(2), matcher.end(2));
            return apiResponse.withGenericError(error.toBuilder()
                    .customType(customErrorType)
                    .customComment(customErrorComment)
                    .build());
        } else {
            return apiResponse;
        }
    }

    public AeroflotOrderCreateResult initPayment(@NonNull AeroflotVariant variant,
                                                 @NonNull List<TravellerInfo> travellers, @NonNull String tokenizedCard,
                                                 @NonNull ClientInfo clientInfo, @NonNull String redirectUrl,
                                                 @NonNull RpcContext context) {
        AeroflotRequestContext bookingContext = variant.getContext();
        AirShoppingRs airShoppingRs = v3CompatibilityConverter.convertToV3AirShoppingDataForPriceCheck(variant);
        OrderCreateRq orderCreateRq = requestFactory.createOrderCreateRq(OrderCreateRequestParams.builder()
                .language(bookingContext.getLanguage())
                .countryOfSale(bookingContext.getCountryCode())
                .dataLists(airShoppingRs.getResponse().getDataLists())
                .offer(AeroflotNdcApiV3Helper.findOfferById(airShoppingRs, variant.getOffer().getId()))
                .travellers(travellers)
                .contactEmail(clientInfo.getEmail())
                .contactPhoneCountryCode(clientInfo.getPhoneCountryCode())
                .contactPhoneNumber(clientInfo.getPhoneNumber())
                .tokenizedCard(tokenizedCard)
                .redirectUrl(redirectUrl)
                .build());

        String requestXml = xmlConverter.convertToXml(orderCreateRq);
        HttpClient.ResponseWithStatus responseXmlAndStatus;
        try {
            responseXmlAndStatus = getHttpResponseWithStatus(true, requestXml, context
                    .setExtraHeaders(ImmutableMap.of(
                            "X-Forwarded-For", clientInfo.getUserIp(),
                            "User-Agent", config.getUserAgent()
                    )));
            if (responseXmlAndStatus.status() == 502 && responseXmlAndStatus.response() == "Backend unavailable\n") {
                throw new BookingRetryableException("NDC API is temporarily unavailable: backend unavailable");
            } else if (responseXmlAndStatus.status() == 401 || responseXmlAndStatus.status() == 403) {
                throw new BookingFailureException(BookingFailureReason.PARTNER_API_TEMPORARY_UNAVAILABLE,
                        "NDC API is temporarily unavailable: access denied");
            }
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new RuntimeException(e);
        } catch (IOException | TimeoutException e) {
            if (e instanceof UnknownHostException || e instanceof ConnectException) {
                // these exceptions mean that there wasn't any actual call because of network unavailability
                throw new BookingRetryableException("OrderCreateRQ call hasn't happened: " + e.getMessage(), e);
            }
            throw new RuntimeException(e);
        }

        OrderViewRs orderCreateRs = convertFromXmlAndValidate(responseXmlAndStatus.response(), OrderViewRs.class);
        handleOrderCreateErrors(orderCreateRs, variant.getOffer().getId());

        AeroflotOrderCreateResult createResult = extractAeroflotOrderCreateResult(orderCreateRs);
        handleOrderCreatePaymentErrors(createResult);
        return createResult;
    }

    private AeroflotOrderRef extractOrderRef(OrderViewRs orderViewRs) {
        String pnr = null;
        String orderId = orderViewRs.getResponse().getOrder().getOrderID();
        List<TicketDocInfo> tickets = orderViewRs.getResponse().getTicketDocInfo();
        if (tickets != null && !tickets.isEmpty()) {
            Set<String> pnrSet = tickets.stream().map(t -> t.getBookingRef().getBookingID()).collect(toSet());
            Preconditions.checkArgument(pnrSet.size() == 1, "Unexpected pnr set: %s", pnrSet);
            pnr = pnrSet.iterator().next();
        }
        if (pnr == null) {
            pnr = AeroflotNdcApiV3Helper.extractPnrFromOrderIdSafe(orderId);
        }
        return AeroflotOrderRef.builder()
                .pnr(pnr)
                // no such pnr date in API v3, will use fuzzy search for confirmation messages from Sabre MQ
                .pnrDate("PNR_date_" + Instant.now().atZone(PNR_DATE_TIME_ZONE_ID).toLocalDate())
                .orderId(orderId)
                .mdOrderId(null)
                .build();
    }

    private AeroflotOrderCreateResult extractAeroflotOrderCreateResult(OrderViewRs orderViewRs) {
        AeroflotOrderStatus orderStatus = AeroflotOrderStatus.forValue(
                orderViewRs.getPaymentInfo().getPaymentTrx().getDescText());
        String subStatusTxt = orderViewRs.getPaymentInfo().getPaymentTrx().getTrxDataText();
        AeroflotOrderSubStatus orderSubStatus = Strings.isNullOrEmpty(subStatusTxt) ? null :
                AeroflotOrderSubStatus.forValue(subStatusTxt);
        AeroflotOrderRef orderRef = extractOrderRef(orderViewRs);
        PaymentCard paymentCard = orderViewRs.getPaymentInfo().getPaymentMethod().getPaymentCard();
        PaymentInstructions payment3dsInstructions = paymentCard != null ?
                paymentCard.getSecurePayerAuthenticationInstructions() : null;
        String confirmationUrl = payment3dsInstructions != null ? payment3dsInstructions.getRedirectionURL() : null;
        Map<String, List<AeroflotTicketCoupon>> couponStatusCodes = extractAeroflotCouponStatusCodes(orderViewRs);
        return AeroflotOrderCreateResult.builder()
                .orderRef(orderRef)
                .statusCode(orderStatus)
                .subStatusCode(orderSubStatus)
                .couponStatusCodes(couponStatusCodes)
                .confirmationUrl(confirmationUrl)
                .build();
    }

    private Map<String, List<AeroflotTicketCoupon>> extractAeroflotCouponStatusCodes(OrderViewRs orderViewRs) {
        var docInfo =
                Optional.ofNullable(orderViewRs).map(OrderViewRs::getResponse).map(OrderViewRsBody::getTicketDocInfo).orElse(null);
        if (docInfo == null) {
            return new HashMap<>();
        }
        return docInfo.stream()
                .filter(t -> t.getTicket().getTicketDocTypeCode() == AeroflotTicketDocTypeCode.TICKET)
                .collect(Collectors.toMap(t -> t.getTicket().getTicketNumber(),
                                t -> t.getTicket().getCoupon().stream()
                                        .map(c -> new AeroflotTicketCoupon(c.getCouponNumber(),
                                                c.getCouponStatusCode()))
                                        .collect(toList())
                        )
                );
    }

    private void handleOrderCreateErrors(OrderViewRs orderCreateRs, @NonNull String offerId) {
        if (orderCreateRs.getError() == null) {
            return;
        }
        String errorCode = orderCreateRs.getError().getError().getCode();
        String errorMessage = orderCreateRs.getError().getError().getDescText();
        if (!Strings.isNullOrEmpty(errorCode)) {
            switch (AeroflotNdcApiErrorCode.getCode(errorCode, errorMessage)) {
                case TEMPORARILY_UNAVAILABLE:
                    throw new BookingFailureException(BookingFailureReason.PARTNER_API_TEMPORARY_UNAVAILABLE,
                            "NDC API is temporarily unavailable: " + errorMessage);
                case NOT_AVAILABLE:
                case SEGMENT_DEPARTURE_IS_TOO_CLOSE:
                case SABRE_FLIGHT_NO_OP:
                    throw new VariantNotAvailableException("The offer isn't available anymore: offer_id=" + offerId + ", error_message=" + errorMessage);
                case TARIFF_NOT_AVAILABLE:
                case FARE_NOT_AVAILABLE:
                    throw new VariantNotAvailableException("The offer fare isn't available anymore: offer_id=" + offerId + ", error_message=" + errorMessage);
                case INVALID_AMOUNT:
                    throw new PriceChangedException("The offer price has changed: offer_id=" + offerId + ", " +
                            "error_message=" + errorMessage);
                case INVALID_CONTACT:
                    // todo(tlg-13): need to think about private data masking
                    throw new BookingFailureException(BookingFailureReason.INVALID_CONTACT,
                            "Client contact is invalid: offer_id=" + offerId + ", error_message=" + errorMessage);
            }
            throw new RuntimeException("Unexpected OrderCreateRQ error: " + orderCreateRs.getError());
        }

        StatusMessage warning = orderCreateRs.getResponse().getWarning();
        if (warning != null) {
            log.info("Order create result comes with a warning: code={}, description={}",
                    warning.getDescText(), warning.getDescText());
        }
    }

    private void handleOrderCreatePaymentErrors(AeroflotOrderCreateResult result) {
        // we can't be sure that these errors can be safely re-tried
        //handleCommonErrors(rsp);

        if (result.isPaid() || result.is3dsRequired()) {
            // no payment errors
            return;
        }
        AeroflotOrderRef orderRef = result.getOrderRef();
        log.info("Received a valid BookingReference: {}", orderRef);

        if (result.getSubStatusCode().isCardRejectedError()) {
            throw new AeroflotPaymentException(
                    "Order payment error: card has been rejected",
                    PaymentFailureReason.PAYMENT_REJECTED,
                    orderRef
            );
        }

        // semi-failed orders: the returned pnr is stored (if any) and the order is cancelled,
        // the user can still complete the payment via the reminder email from Aeroflot
        throw new AeroflotPaymentException("Order payment error", PaymentFailureReason.OTHER, orderRef);
    }

    @NonNull
    public AeroflotOrderCreateResult getOrderStatus(@NonNull AeroflotServicePayload payload,
                                                    @NonNull RpcContext context) {
        return getOrderStatusImpl(payload.getVariant(), payload.getBookingRef(), payload.getClientInfo(), context);
    }

    @NonNull
    private AeroflotOrderCreateResult getOrderStatusImpl(@NonNull AeroflotVariant variant,
                                                         @NonNull AeroflotOrderRef orderRef,
                                                         @NonNull ClientInfo clientInfo, @NonNull RpcContext context) {
        OrderRetrieveRq orderRetrieveRq = requestFactory.createOrderRetrieveRq(OrderRetrieveRequestParams.builder()
                .language(variant.getContext().getLanguage())
                .orderId(orderRef.getOrderId())
                .ownerCode(variant.getOffer().getOwnerCode())
                .build());
        String xmlReq = xmlConverter.convertToXml(orderRetrieveRq);
        String response;
        try {
            response = sendHttpRequest(true, xmlReq, context
                    .setExtraHeaders(ImmutableMap.of(
                            "X-Forwarded-For", clientInfo.getUserIp(),
                            "User-Agent", config.getUserAgent()
                    )));
        } catch (IOException | TimeoutException | InterruptedException e) {
            if (e instanceof InterruptedException) {
                Thread.currentThread().interrupt();
            }
            throw new BookingRetryableException("Recoverable error during order status OrderViewRQ; " +
                    "e.msg=" + e.getMessage(), e);
        }
        OrderViewRs orderViewRs = convertFromXmlAndValidate(response, OrderViewRs.class);

        if (orderViewRs.getError() != null) {
            throw new RuntimeException("Unexpected OrderRetrieveRQ error: " + orderViewRs.getError());
        }

        return extractAeroflotOrderCreateResult(orderViewRs);
    }

    @NonNull
    private String extractNotEmptyText(@NonNull JsonNode node, @NonNull String field) {
        JsonNode v = node.path(field);
        String value = v.asText(null);
        Preconditions.checkArgument(!Strings.isNullOrEmpty(value), "a not empty %s field value is expected", field);
        return value;
    }
}
