package ru.yandex.avia.booking.partners.gateways.aeroflot;

import java.util.Map;
import java.util.stream.Stream;

import com.google.common.base.Strings;
import com.google.common.collect.ImmutableMap;
import lombok.AllArgsConstructor;
import lombok.Getter;

import static java.util.stream.Collectors.toMap;

@AllArgsConstructor
@Getter
public enum AeroflotNdcApiErrorCode {

    // <Error ShortText="Invalid Place of Destination Code" Code="101" Type="101">Аэропорт прибытия недоступен для бронирования: KZO</Error>
    INVALID_PLACE("101"),
    // <Error ShortText="Invalid/Missing Departure Date" Code="102" Type="102">Дата вылета в прошлом: 2020-05-15</Error>
    INVALID_DEPARTURE_DATE("102"),

    // <Error ShortText="System Temporarily Unavailable" Code="304" Type="304">Произошла ошибка. Попробуйте позже или обратитесь в контактный центр ПАО «Аэрофлот». При письменном обращении, пожалуйста, укажите код ошибки 'sb1-910c6c6dfa49844b22317af6c742f52c0e185743' в тексте сообщения.</Error>
    // <Error ShortText="System Temporarily Unavailable" Code="304" Type="304">Ошибка связи</Error>
    // <Error ShortText="System Temporarily Unavailable" Code="304" Type="304">Критическая ошибка</Error>
    TEMPORARILY_UNAVAILABLE("304"),

    // <Error ShortText="Contact element (phone and/or address) invalid" Code="317" Type="317">Неверный формат email: ORXXXXXXXXXYA_@LIST.RU799999992</Error>
    INVALID_CONTACT("317"),

    // <Error ShortText="Segment departure is too close" Code="350" Type="350">Дата вылета меньше допустимого значения: 2019-07-31 08:50:00+04:00</Error>
    SEGMENT_DEPARTURE_IS_TOO_CLOSE("350"),

    // <Error ShortText="Not Available and Waitlist is Closed" Code="421" Type="421">Свободные места с выбранной стоимостью закончились</Error>
    NOT_AVAILABLE("421"),

    // <Error ShortText="Sabre fare not guaranteed" Code="71I" Type="71I">Выбранный тариф недоступен</Error>
    FARE_NOT_AVAILABLE("71I"),

    // <Error ShortText="Invalid amount" Code="727" Type="727"/>
    INVALID_AMOUNT("727"),

    // <Error ShortText="Rate not available" Code="740" Type="740">Выбранный тариф недоступен</Error>
    TARIFF_NOT_AVAILABLE("740"),

    // <Error ShortText="Unable to process - system error" Code="911" Type="911">Сервер бронирования недоступен</Error>
    // <Error ShortText="Unable to process - system error" Code="911" Type="911">Номер Брони не известен системе бронирования</Error>
    // <Error ShortText="Unable to process - system error" Code="911" Type="911">SBService error: HTTP 500 1001000211 SabrePNRNotFoundException</Error>
    UNABLE_TO_PROCESS("911"),

    // <Error ShortText="Unable to process - system error" Code="911" Type="911">SBService error: HTTP 500 1001000260 SabreFlightNoOpException</Error>
    SABRE_FLIGHT_NO_OP("911_sabre_1001000260"),

    // for all not mapped codes
    UNKNOWN("<unknown>");


    private static final Map<String, AeroflotNdcApiErrorCode> codeIndex = ImmutableMap.copyOf(
            Stream.of(AeroflotNdcApiErrorCode.values()).collect(toMap(AeroflotNdcApiErrorCode::getCode, e -> e))
    );

    // there could semi-textual codes like "74W"
    private final String code;

    public static AeroflotNdcApiErrorCode getCode(String value, String message) {
        AeroflotNdcApiErrorCode messageBasedCode = detectMessageBasedCode(message);
        if (messageBasedCode != null) {
            return messageBasedCode;
        }
        return codeIndex.getOrDefault(value, UNKNOWN);
    }

    private static AeroflotNdcApiErrorCode detectMessageBasedCode(String message) {
        if (Strings.isNullOrEmpty(message)) {
            return null;
        }
        if (message.contains("1001000260") || message.contains("SabreFlightNoOpException")) {
            return SABRE_FLIGHT_NO_OP;
        }
        return null;
    }
}
